<?php

/** ******************************************
 * LiteSpeed Web Server Plugin for WHM
 *
 * @author LiteSpeed Technologies, Inc. (https://www.litespeedtech.com)
 * @copyright (c) 2018-2020
 * ******************************************* */

namespace LsPanel;

use \Lsc\Wp\LSCMException;
use \Lsc\Wp\Panel\ControlPanel;
use \Lsc\Wp\Panel\CPanel;

class WhmMod_LiteSpeed_CPanelConf
{

    /**
     * @var string
     */
    const CPANEL_PLUGIN_DIR = '/usr/local/cpanel/base/frontend/paper_lantern/ls_web_cache_manager/';

    /**
     * @var string
     */
    const CPANEL_PLUGIN_CONF = '/usr/local/cpanel/base/frontend/paper_lantern/ls_web_cache_manager/data/lswcm.conf';

    /**
     * @since 4.1
     * @var string
     */
    const CPANEL_PLUGIN_TEMP_CONF = __DIR__ . '/data/lswcm.conf';

    /**
     * @var string
     */
    const CPANEL_PLUGIN_THEME_DIR = '/usr/local/cpanel/base/frontend/paper_lantern/ls_web_cache_manager/landing';

    /**
     * @var string
     */
    const FLD_CPANEL_PLUGIN_INSTALLED = 'cpanel_plugin_installed';

    /**
     * @var string
     */
    const FLD_CPANEL_PLUGIN_AUTOINSTALL = 'cpanel_plugin_autoinstall';

    /**
     * @var string
     */
    const FLD_LSWS_DIR = 'lsws_dir';

    /**
     * @var string
     */
    const FLD_VHOST_CACHE_ROOT = 'vhost_cache_root';

    /**
     * @var string
     */
    const FLD_USE_CUST_TPL = 'use_cust_tpl';

    /**
     * @var string
     */
    const FLD_CUST_TPL_NAME = 'cust_tpl_name';

    /**
     * @since 4.1
     * @var string
     */
    const FLD_GENERATE_EC_CERTS = 'generateEcCerts';

    /**
     * @since 4.1
     * @var string
     */
    const SETTING_OFF = 0;

    /**
     * @since 4.1
     * @var string
     */
    const SETTING_ON = 1;

    /**
     * @since 4.1
     * @var string
     */
    const SETTING_ON_PLUS_AUTO = 2;

    /**
     * @var ControlPanel
     */
    private $panelEnv;

    /**
     * @var mixed[]
     */
    private $data;

    /**
     * @var string[]
     */
    private $succ_msgs = array();

    /**
     * @var string[]
     */
    private $err_msgs = array();

    /**
     * @var boolean
     */
    private $save = false;

    /**
     *
     * @throws LSCMException  Thrown indirectly.
     */
    public function __construct()
    {
        $this->init();
    }

    /**
     *
     * @return void
     * @throws LSCMException  Thrown indirectly.
     */
    private function init()
    {
        $this->panelEnv =
            ControlPanel::getClassInstance('\Lsc\Wp\Panel\CPanel');

        $this->populateSettingDefaultValues();

        $this->data[self::FLD_CPANEL_PLUGIN_AUTOINSTALL] =
                (int)CPanel::isCpanelPluginAutoInstallOn();

        $this->data[self::FLD_CPANEL_PLUGIN_INSTALLED] =
            file_exists(self::CPANEL_PLUGIN_DIR);

        $contents = $this->getConfFileContent();

        if ( $contents == '' ) {
            $this->writeDefaultConf();
            return;
        }

        $this->loadSettingLswsHomeDir($contents);
        $this->loadSettingVhCacheRoot($contents);
        $this->loadSettingGenerateEcCerts($contents);
        $this->loadSettingUseCustTpl($contents);
        $this->loadSettingCustTpl($contents);

        /**
         * Save any detected changes while loading.
         */
        $this->trySaveConf();
    }

    /**
     *
     * @since 4.1
     */
    private function populateSettingDefaultValues()
    {
        $this->data[self::FLD_LSWS_DIR] = '';
        $this->data[self::FLD_VHOST_CACHE_ROOT] = '';
        $this->data[self::FLD_GENERATE_EC_CERTS] = self::SETTING_OFF;
        $this->data[self::FLD_USE_CUST_TPL] = false;
        $this->data[self::FLD_CUST_TPL_NAME] = '';
    }

    /**
     *
     * @since 4.1
     *
     * @return string
     */
    private function getConfFileContent()
    {
        $ret = '';

        if ( file_exists(self::CPANEL_PLUGIN_CONF)
                && ($content = file_get_contents(self::CPANEL_PLUGIN_CONF)) ) {

            $ret = $content;
        }
        elseif ( file_exists(self::CPANEL_PLUGIN_TEMP_CONF)
            && ($content = file_get_contents(self::CPANEL_PLUGIN_TEMP_CONF)) ) {

            $ret = $content;
        }

        return $ret;
    }

    private function writeDefaultConf()
    {
        if ( defined('LSWS_HOME') ) {
            $this->data[self::FLD_LSWS_DIR] = LSWS_HOME;
        }

        $vhCacheRoot = $this->panelEnv->getVHCacheRoot();

        if ( $vhCacheRoot != ControlPanel::NOT_SET ) {
            $this->data[self::FLD_VHOST_CACHE_ROOT] = $vhCacheRoot;
        }

        $this->save = true;
        $this->trySaveConf();
    }

    /**
     *
     * @param string  $pattern
     * @param string  $contents
     * @return string
     */
    private function readSetting( $pattern, $contents )
    {
        preg_match($pattern, $contents, $matches);

        if ( isset($matches[1]) ) {
            return $matches[1];
        }
        else {
            return '';
        }
    }

    /**
     * Attempts to create a temporary WhmMod_LiteSpeed_CPanelConf object which
     * should create a cPanel config file if one does not exist or, if it does
     * exist, read the config file setting values and update the file if
     * changes are detected.
     */
    public static function verifyCpanelPluginConfFile()
    {
        new self();
    }

    /**
     *
     * @param string  $contents  Contents of lswcm.conf file.
     */
    private function loadSettingLswsHomeDir( $contents )
    {
        $pattern = '/LSWS_HOME_DIR = "(.*)"/';
        $setting = $this->readSetting($pattern, $contents);
        $this->data[self::FLD_LSWS_DIR] = $setting;

        if ( defined('LSWS_HOME')
                && $this->data[self::FLD_LSWS_DIR] != LSWS_HOME) {

            $this->setLswsDir(LSWS_HOME);
        }
    }

    /**
     *
     * @param string  $contents  Contents of lswcm.conf file.
     */
    private function loadSettingVhCacheRoot( $contents )
    {
        $pattern = '/VHOST_CACHE_ROOT = "(.*)"/';
        $setting = $this->readSetting($pattern, $contents);
        $this->data[self::FLD_VHOST_CACHE_ROOT] = $setting;

        $vhCacheRoot = $this->panelEnv->getVHCacheRoot();

        if ( $vhCacheRoot == ControlPanel::NOT_SET ) {
            $vhCacheRoot = '';
        }

        if ( $this->data[self::FLD_VHOST_CACHE_ROOT] != $vhCacheRoot ) {
            $this->setVhCacheRoot($vhCacheRoot);
        }
    }

    /**
     *
     * @param string  $contents  Contents of lswcm.conf file.
     */
    private function loadSettingUseCustTpl( $contents )
    {
        $pattern = '/USE_CUST_TPL = (\d)/';
        $setting = $this->readSetting($pattern, $contents);
        $this->data[self::FLD_USE_CUST_TPL] = (bool)$setting;
    }

    /**
     *
     * @param string  $contents  Contents of lswcm.conf file.
     */
    private function loadSettingCustTpl( $contents )
    {
        $pattern = '/CUST_TPL = "(.*)"/';
        $setting = $this->readSetting($pattern, $contents);
        $this->data[self::FLD_CUST_TPL_NAME] = $setting;
    }

    /**
     *
     * @since 4.1
     *
     * @param string  $contents  Contents of lswcm.conf file.
     */
    private function loadSettingGenerateEcCerts( $contents )
    {
        $pattern = '/GENERATE_EC_CERTS = (\d)/';
        $setting = $this->readSetting($pattern, $contents);
        $this->data[self::FLD_GENERATE_EC_CERTS] = (int)$setting;
    }

    /**
     *
     * @param string  $field
     * @return mixed
     */
    public function getData( $field = '' )
    {
        if ( !isset($this->data[$field]) ) {
            return null;
        }

        return $this->data[$field];
    }

    /**
     *
     * @param boolean  $clear
     * @return string[]
     */
    public function getSuccMsgs( $clear = false )
    {
        $succMsgs = $this->succ_msgs;

        if ( $clear ) {
            $this->succ_msgs = array();
        }

        return $succMsgs;
    }

    /**
     *
     * @param boolean  $clear
     * @return string[]
     */
    public function getErrMsgs( $clear = false )
    {
        $errMsgs = $this->err_msgs;

        if ( $clear ) {
            $this->err_msgs = array();
        }

        return $errMsgs;
    }

    public function setAutoInstallUse( $autoInstall )
    {
        if ( $autoInstall != $this->data[self::FLD_CPANEL_PLUGIN_AUTOINSTALL] ) {
            $this->data[self::FLD_CPANEL_PLUGIN_AUTOINSTALL] = $autoInstall;
            $this->save = true;
        }

        if ( $autoInstall === 1 ) {
            CPanel::turnOnCpanelPluginAutoInstall();
        }
        else {
            CPanel::turnOffCpanelPluginAutoInstall();
        }

        return true;
    }

    /**
     *
     * @param boolean  $useCustTpl
     * @return boolean
     */
    public function setTplUse( $useCustTpl )
    {
        if ( $useCustTpl != $this->data[self::FLD_USE_CUST_TPL] ) {
            $this->data[self::FLD_USE_CUST_TPL] = $useCustTpl;
            $this->save = true;
        }

        return true;
    }

    public function clearTplName()
    {
        $this->data[self::FLD_CUST_TPL_NAME] = '';
        $this->save = true;
    }

    /**
     *
     * @param string  $newCustTpl
     * @return boolean
     */
    public function setTplName( $newCustTpl )
    {
        if ( $newCustTpl == '' ) {
            return false;
        }
        else if ( $newCustTpl == $this->data[self::FLD_CUST_TPL_NAME] ) {
            return true;
        }

        $themeDir = self::CPANEL_PLUGIN_THEME_DIR;

        $newCustTplSafe = htmlspecialchars($newCustTpl);

        if ( file_exists("{$themeDir}/{$newCustTpl}/index.php") ) {

            $this->data[self::FLD_CUST_TPL_NAME] = $newCustTpl;
            $this->save = true;

            $this->succ_msgs[] = "Custom template {$newCustTplSafe} set";

            return true;
        }
        else {
            $this->err_msgs[] =
                'Could not find index.php file for custom template '
                    . "'{$newCustTplSafe}'";

            return false;
        }
    }

    /**
     *
     * @param string  $newLswsDir
     * @return boolean
     */
    private function setLswsDir( $newLswsDir )
    {
        if ( !is_dir($newLswsDir) ) {
            return false;
        }

        $this->data[self::FLD_LSWS_DIR] = $newLswsDir;
        $this->save = true;

        $this->succ_msgs[] = 'LSWS_HOME_DIR set in cPanel Plugin conf file.';

        return true;
    }

    /**
     *
     * @param string  $newVhCacheRoot
     * @return boolean
     */
    public function setVhCacheRoot( $newVhCacheRoot )
    {
        $this->data[self::FLD_VHOST_CACHE_ROOT] = $newVhCacheRoot;
        $this->save = true;

        $this->succ_msgs[] = 'VHOST_CACHE_ROOT set in cPanel Plugin conf file.';

        return true;
    }

    /**
     *
     * @since 4.1
     *
     * @param int  $generateEcCerts
     * @return boolean
     */
    public function setGenerateEcCerts( $generateEcCerts )
    {
        if ( $generateEcCerts < 0 || $generateEcCerts > 2 ) {
            return false;
        }

        if ( $generateEcCerts != $this->data[self::FLD_GENERATE_EC_CERTS] ) {
            $this->data[self::FLD_GENERATE_EC_CERTS] = $generateEcCerts;
            $this->save = true;
        }

        return true;
    }

    /**
     *
     * @since 4.1
     *
     * @return bool
     */
    private function tryRunCertSupportScripts()
    {
        if ( !file_exists(self::CPANEL_PLUGIN_DIR) ) {
            return false;
        }

        if ( $this->data[self::FLD_GENERATE_EC_CERTS] == self::SETTING_OFF ) {
            exec(self::CPANEL_PLUGIN_DIR . '/scripts/cert_support_remove.sh');
        }
        else {
            exec(self::CPANEL_PLUGIN_DIR . '/scripts/cert_support_add.sh');
        }

        return true;
    }

    /**
     *
     * @return void
     */
    public function trySaveConf()
    {
        if ( !$this->save ) {
            return;
        }

        $useCustTpl = (int)$this->data[self::FLD_USE_CUST_TPL];

        $content = <<<EOF
LSWS_HOME_DIR = "{$this->data[self::FLD_LSWS_DIR]}"
VHOST_CACHE_ROOT = "{$this->data[self::FLD_VHOST_CACHE_ROOT]}"
USE_CUST_TPL = {$useCustTpl}
CUST_TPL = "{$this->data[self::FLD_CUST_TPL_NAME]}"
GENERATE_EC_CERTS = {$this->data[self::FLD_GENERATE_EC_CERTS]}

EOF;

        file_put_contents(self::CPANEL_PLUGIN_TEMP_CONF, $content);

        if ( $this->data[self::FLD_CPANEL_PLUGIN_INSTALLED] ) {
            file_put_contents(self::CPANEL_PLUGIN_CONF, $content);
        }

        $this->save = false;

        $this->tryRunCertSupportScripts();
    }

}
