#!/bin/sh
#set -x
#set -o errexit

# /********************************************************************
# LiteSpeed EA4 TimeZoneDB Auto Installer
# @Author: LiteSpeed Technologies, Inc. (https://www.litespeedtech.com)
# @Copyright: (c) 2016-2019
# @Version: 1.2.5
# *********************************************************************/

show_help()
{
    cat <<EOF

    # ***********************************************************************
    #
    # This script will auto build the source version of timezonedb.so for
    # all instances of EA4 PHP.
    #
    # Prerequisites:
    #    1. cPanel's EA PHP environment only.
    #
    # Command:
    #    ./ea4.sh
    #
    # Example:
    #    ./ea4.sh y true
    #
    # Input parameters:
    #
    # 1. (Optional) YN
    #    Auto build it without input.
    #    Only accepts "y" or "X.X" or "X.X.X", X being integers.
    #
    # 2. (Optional) check
    #    Checks all versions and exits.
    #    Valid input is "true".
    #
    # ***********************************************************************

EOF
exit 1
}

if [[ $# -gt 0 ]] ; then
    if [[ "$1" = "y" ]] ; then
        yn="y"
    elif [[ "$1" =~ ([0-9]+\.)?[0-9]+\.[0-9]+ ]] ; then
        yn="$1"
    else
        echo "Invalid parameter."
        show_help
    fi
else
    yn=false
fi

if [[ $# -gt 1 ]]; then
    if [[ "$2" = "true" ]]; then
        check=true
    fi
fi

cagefs()
{
    echo "Checking to see if CageFS is enabled."
    if [[ -e "/usr/sbin/cagefsctl" ]]; then
        if /usr/sbin/cagefsctl --cagefs-status | grep -q "Enabled"; then
            echo "CageFS found and enabled."
            echo "Updating CageFS to support the new module."
            /usr/sbin/cagefsctl --force-update &>/dev/null
            if [ $? -eq 0 ]; then
                echo "CageFS updated successfully."
            else
                echo "**ERROR** Error running 'cagefsctl --force-update'. Please try running it manually."
            fi
        else
            echo "CageFS not enabled."
            echo "Skipping."
        fi
    else
        echo "CageFS not installed."
        echo "Skipping."
    fi
}

# Error Handling
checkErrs()
{
    if [[ "${1}" -ne "0" ]] ; then
        if [[ "$BACKUPMODULE" = true ]]; then
            cp ""$MODULEPATH"/timezonedb.so."$NOW"" ""$MODULEPATH"/timezonedb.so"
        fi
        if [[ "$BACKUPINI" = true ]]; then
            cp ""$INIPATH"/php.ini."$NOW"" ""$INIPATH"/php.ini"
        fi
        echo "**ERROR** EA4 PHP ""$PHPVERSION"" ${2}"
        exit ${1}
    fi
}

check_devel() {
  rpm -qa | grep ea-php${MMVERSION}-php-devel &> /dev/null
  if [[ $? -ne 0 ]]; then
    checkErrs 1 "ea-php${MMVERSION}-php-devel was not found. This is required to build TimeZomeDB. Please run 'yum install -y ea-php${MMVERSION}-php-devel' to fix this issue."
  fi
}

if [[ "$yn" = false ]]; then
    read -p "Do you wish to install timezonedb for all Easy Apache 4 PHP Versions? (y, n, x.x, or x.x.x) " yn
    read -p "Do you wish to check timezonedb for all Easy Apache 4 PHP Versions? (true or false) " check
fi

echo ""
echo ""

if [[ "$yn" = "y" ]]; then
    EAV=("5.4" "5.5" "5.6" "7.0" "7.1" "7.2" "7.3")
    for EAVERSION in ${EAV[@]}
    do
        MMVERSION="${EAVERSION//.}"
        echo "Checking to see if Easy Apache 4 PHP "$EAVERSION" is installed....."
        if [[ -e "/opt/cpanel/ea-php"$MMVERSION"/root/usr/bin/phpize" ]]; then
            if [[ "$check" = true ]]; then
                ./buildtimezone.sh "$EAVERSION" y
            else
                check_devel
                ./buildtimezone.sh "$EAVERSION" ignorecagefs
            fi
        else
            echo "EA4 PHP "$EAVERSION" not installed!"
        fi
        echo ""
        echo ""
    done
    echo ""
    echo ""
    echo "TimeZoneDB installed for all Easy Apache 4 PHP Versions."
elif [[ "$yn" =~ ([0-9]+\.)?[0-9]+\.[0-9]+ ]]; then
    MMVERSION="${yn//.}"
    if [[ "$check" = true ]]; then
        ./buildtimezone.sh "$yn" y
    else
        check_devel
        ./buildtimezone.sh "$yn" ignorecagefs
    fi
    echo ""
    echo ""
    echo "TimeZoneDB installed for all Easy Apache 4 PHP Versions."
else
    echo "You have cancelled the install."
    show_help
fi

if [[ "$check" != true ]]; then
  cagefs
fi

exit 0
