<?php
/**
 * includes/image_utils.php
 * Utilities for high-quality image processing using Imagick (if available) with GD fallback.
 * Generates:
 *  - Original (kept as-is)
 *  - 226x226 square thumbnail (center-crop or pad-to-square if requested)
 *  - 10x10 tiny placeholder (used as LQIP for lazy-loading)
 *
 * All functions return associative arrays with 'ok', 'msg', and any paths produced.
 */

declare(strict_types=1);

if (!defined('IMGUTILS_INCLUDED')) {
    define('IMGUTILS_INCLUDED', true);
}

function imgutils_supported(): array {
    return [
        'imagick' => class_exists('Imagick'),
        'gd'      => function_exists('imagecreatetruecolor'),
    ];
}

function _ensure_dir(string $absDir): bool {
    if (is_dir($absDir)) return true;
    return @mkdir($absDir, 0775, true);
}

function _ext_from_mime(string $mime): string {
    switch (strtolower($mime)) {
        case 'image/jpeg':
        case 'image/jpg': return 'jpg';
        case 'image/png': return 'png';
        case 'image/gif': return 'gif';
        case 'image/webp': return 'webp';
        default: return 'jpg';
    }
}

function imgutils_save_upload(array $file, string $absDir, string $relDir): array {
    if (empty($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
        return ['ok'=>false, 'msg'=>'No upload'];
    }
    $info = getimagesize($file['tmp_name']);
    if (!$info) return ['ok'=>false, 'msg'=>'Invalid image'];
    $ext = _ext_from_mime($info['mime'] ?? '');
    if (!_ensure_dir($absDir)) return ['ok'=>false, 'msg'=>'Cannot create dir'];
    $name = uniqid('img_', true) . ".$ext";
    $abs  = rtrim($absDir,'/')."/".$name;
    $rel  = rtrim($relDir,'/')."/".$name;
    if (!move_uploaded_file($file['tmp_name'], $abs)) {
        return ['ok'=>false, 'msg'=>'Failed to move upload'];
    }
    @chmod($abs, 0664);
    return ['ok'=>true, 'abs'=>$abs, 'rel'=>$rel, 'width'=>$info[0]??0, 'height'=>$info[1]??0, 'mime'=>$info['mime']??''];
}

/**
 * Pad to square with white background (RGB 255).
 */
function imgutils_pad_to_square(string $srcAbs, string $dstAbs, string $bgHex = '#FFFFFF'): array {
    $support = imgutils_supported();
    if ($support['imagick']) {
        try {
            $im = new Imagick($srcAbs);
            $w = $im->getImageWidth();
            $h = $im->getImageHeight();
            $size = max($w, $h);
            $canvas = new Imagick();
            $canvas->newImage($size, $size, new ImagickPixel($bgHex), $im->getImageFormat());
            $x = intval(($size - $w) / 2);
            $y = intval(($size - $h) / 2);
            $canvas->compositeImage($im, Imagick::COMPOSITE_DEFAULT, $x, $y);
            $canvas->setImageCompressionQuality(92);
            $canvas->writeImage($dstAbs);
            $im->destroy(); $canvas->destroy();
            return ['ok'=>true];
        } catch (Throwable $e) {
            return ['ok'=>false, 'msg'=>$e->getMessage()];
        }
    } elseif ($support['gd']) {
        $info = getimagesize($srcAbs);
        if (!$info) return ['ok'=>false, 'msg'=>'Bad image'];
        [$w, $h] = $info;
        $size = max($w, $h);
        $dst = imagecreatetruecolor($size, $size);
        $white = imagecolorallocate($dst, 255, 255, 255);
        imagefilledrectangle($dst, 0, 0, $size, $size, $white);
        switch ($info['mime']) {
            case 'image/png': $src = imagecreatefrompng($srcAbs); break;
            case 'image/gif': $src = imagecreatefromgif($srcAbs); break;
            case 'image/webp': $src = function_exists('imagecreatefromwebp') ? imagecreatefromwebp($srcAbs) : null; break;
            default: $src = imagecreatefromjpeg($srcAbs);
        }
        if (!$src) return ['ok'=>false, 'msg'=>'Unsupported format'];
        $x = intval(($size - $w) / 2);
        $y = intval(($size - $h) / 2);
        imagecopy($dst, $src, $x, $y, 0, 0, $w, $h);
        $ok = imagejpeg($dst, $dstAbs, 92);
        imagedestroy($src); imagedestroy($dst);
        return ['ok'=>$ok];
    }
    return ['ok'=>false, 'msg'=>'No image library'];
}

/**
 * Crop rectangle (x,y,w,h) from srcAbs to dstAbs
 */
function imgutils_crop(string $srcAbs, string $dstAbs, int $x, int $y, int $w, int $h): array {
    $support = imgutils_supported();
    if ($support['imagick']) {
        try {
            $im = new Imagick($srcAbs);
            $im->cropImage($w, $h, $x, $y);
            $im->setImageCompressionQuality(92);
            $im->writeImage($dstAbs);
            $im->destroy();
            return ['ok'=>true];
        } catch (Throwable $e) { return ['ok'=>false, 'msg'=>$e->getMessage()]; }
    } elseif ($support['gd']) {
        $info = getimagesize($srcAbs);
        if (!$info) return ['ok'=>false, 'msg'=>'Bad image'];
        switch ($info['mime']) {
            case 'image/png': $src = imagecreatefrompng($srcAbs); break;
            case 'image/gif': $src = imagecreatefromgif($srcAbs); break;
            case 'image/webp': $src = function_exists('imagecreatefromwebp') ? imagecreatefromwebp($srcAbs) : null; break;
            default: $src = imagecreatefromjpeg($srcAbs);
        }
        if (!$src) return ['ok'=>false, 'msg'=>'Unsupported format'];
        $dst = imagecreatetruecolor($w, $h);
        imagecopy($dst, $src, 0, 0, $x, $y, $w, $h);
        $ok = imagejpeg($dst, $dstAbs, 92);
        imagedestroy($src); imagedestroy($dst);
        return ['ok'=>$ok];
    }
    return ['ok'=>false, 'msg'=>'No image library'];
}

/**
 * Resize to exact width/height with crop (cover) or contain (pad, if padHex provided).
 */
function imgutils_resize(string $srcAbs, string $dstAbs, int $tw, int $th, bool $cover = true, ?string $padHex = null): array {
    $support = imgutils_supported();
    if ($support['imagick']) {
        try {
            $im = new Imagick($srcAbs);
            $im->setImageCompressionQuality(92);
            $im->setImageInterlaceScheme(Imagick::INTERLACE_JPEG);
            $w = $im->getImageWidth();
            $h = $im->getImageHeight();
            $srcRatio = $w / max(1,$h);
            $dstRatio = $tw / max(1,$th);
            if ($cover) {
                if ($srcRatio > $dstRatio) {
                    $nh = $th;
                    $nw = intval($th * $srcRatio);
                } else {
                    $nw = $tw;
                    $nh = intval($tw / $srcRatio);
                }
                $im->resizeImage($nw, $nh, Imagick::FILTER_LANCZOS, 1);
                $im->cropImage($tw, $th, intval(($nw-$tw)/2), intval(($nh-$th)/2));
            } else {
                $im->resizeImage($tw, $th, Imagick::FILTER_LANCZOS, 1, true);
                if ($padHex) {
                    $canvas = new Imagick();
                    $canvas->newImage($tw, $th, new ImagickPixel($padHex), $im->getImageFormat());
                    $x = intval(($tw - $im->getImageWidth())/2);
                    $y = intval(($th - $im->getImageHeight())/2);
                    $canvas->compositeImage($im, Imagick::COMPOSITE_DEFAULT, $x, $y);
                    $im = $canvas;
                }
            }
            $im->writeImage($dstAbs);
            $im->destroy();
            return ['ok'=>true];
        } catch (Throwable $e) { return ['ok'=>false, 'msg'=>$e->getMessage()]; }
    } elseif ($support['gd']) {
        $info = getimagesize($srcAbs);
        if (!$info) return ['ok'=>false, 'msg'=>'Bad image'];
        [$sw,$sh] = $info;
        $src = null;
        switch ($info['mime']) {
            case 'image/png': $src = imagecreatefrompng($srcAbs); break;
            case 'image/gif': $src = imagecreatefromgif($srcAbs); break;
            case 'image/webp': $src = function_exists('imagecreatefromwebp') ? imagecreatefromwebp($srcAbs) : null; break;
            default: $src = imagecreatefromjpeg($srcAbs);
        }
        if (!$src) return ['ok'=>false, 'msg'=>'Unsupported format'];
        $dst = imagecreatetruecolor($tw,$th);
        if (!$cover && $padHex) {
            // Fill with white
            $r=255;$g=255;$b=255;
            $col=imagecolorallocate($dst,$r,$g,$b);
            imagefilledrectangle($dst,0,0,$tw,$th,$col);
        }
        $srcRatio = $sw / max(1,$sh);
        $dstRatio = $tw / max(1,$th);
        if ($cover) {
            if ($srcRatio > $dstRatio) { // wider
                $nh = $th;
                $nw = intval($th * $srcRatio);
            } else {
                $nw = $tw;
                $nh = intval($tw / $srcRatio);
            }
            $tmp = imagecreatetruecolor($nw,$nh);
            imagecopyresampled($tmp,$src,0,0,0,0,$nw,$nh,$sw,$sh);
            imagecopy($dst,$tmp,intval(($tw-$nw)/2),intval(($th-$nh)/2),intval(($nw-$tw)/2),intval(($nh-$th)/2),$tw,$th);
            imagedestroy($tmp);
        } else {
            if ($srcRatio > $dstRatio) {
                $nw = $tw;
                $nh = intval($tw / $srcRatio);
            } else {
                $nh = $th;
                $nw = intval($th * $srcRatio);
            }
            $tmp = imagecreatetruecolor($nw,$nh);
            imagecopyresampled($tmp,$src,0,0,0,0,$nw,$nh,$sw,$sh);
            imagecopy($dst,$tmp,intval(($tw-$nw)/2),intval(($th-$nh)/2),0,0,$nw,$nh);
            imagedestroy($tmp);
        }
        $ok = imagejpeg($dst,$dstAbs,92);
        imagedestroy($src); imagedestroy($dst);
        return ['ok'=>$ok];
    }
    return ['ok'=>false, 'msg'=>'No image library'];
}

/**
 * Generate all variants (226x226 and 10x10) next to original.
 * Returns paths relative to $relDir.
 */
function imgutils_generate_variants(string $absDir, string $relDir, string $srcRelBasename, bool $padIfNotSquare=true): array {
    $srcAbs = rtrim($absDir,'/')."/".$srcRelBasename;
    $nameNoExt = preg_replace('/\.[^.]+$/','',$srcRelBasename);
    $ext       = strtolower(pathinfo($srcRelBasename, PATHINFO_EXTENSION));
    $thumb226  = $nameNoExt.'_226x226.'.$ext;
    $tiny10    = $nameNoExt.'_10x10.'.$ext;

    $abs226 = rtrim($absDir,'/')."/".$thumb226;
    $abs10  = rtrim($absDir,'/')."/".$tiny10;

    // Ensure square (padding) before resize if requested
    if ($padIfNotSquare) {
        $tmpSquare = rtrim($absDir,'/')."/".$nameNoExt.'_square.'.$ext;
        $pad = imgutils_pad_to_square($srcAbs, $tmpSquare, '#FFFFFF');
        if ($pad['ok']) {
            $srcAbs = $tmpSquare;
        }
    }

    $r1 = imgutils_resize($srcAbs, $abs226, 226, 226, true);
    $r2 = imgutils_resize($srcAbs, $abs10, 10, 10, true);

    return [
        'ok' => ($r1['ok'] ?? false) && ($r2['ok'] ?? false),
        'thumb_226' => rtrim($relDir,'/')."/".$thumb226,
        'tiny_10'   => rtrim($relDir,'/')."/".$tiny10,
    ];
}
