<?php
// add-product.php
include "./includes/Conn.php";
include "./includes/session.php";

/* ---------------- CONFIG: Media Library (local files, no DB) ---------------- */
$LIB_DIR   = __DIR__ . "/uploads/library";     // physical path
$LIB_URL   = "uploads/library";                // public URL prefix (relative to this page)
$TH_DIR    = $LIB_DIR . "/_thumbs";            // thumbs dir
$TH_URL    = $LIB_URL . "/_thumbs";            // thumbs url

// Ensure library + thumbs dir exist
if (!is_dir($LIB_DIR))  { @mkdir($LIB_DIR, 0755, true); }
if (!is_dir($TH_DIR))   { @mkdir($TH_DIR,  0755, true); }

/* ---------------- Helpers ---------------- */

// safer MIME detection (no fatal if fileinfo is missing)
function detect_mime($tmp, $origName='') {
    static $finfoAvail = null;
    if ($finfoAvail === null) $finfoAvail = function_exists('finfo_open');
    $mime = null;

    if ($finfoAvail) {
        $fi = finfo_open(FILEINFO_MIME_TYPE);
        if ($fi) { $mime = @finfo_file($fi, $tmp); finfo_close($fi); }
    } elseif (function_exists('mime_content_type')) {
        $mime = @mime_content_type($tmp);
    } elseif (function_exists('exif_imagetype')) {
        $t = @exif_imagetype($tmp);
        if ($t) $mime = image_type_to_mime_type($t);
    }

    if (!$mime && $origName) {
        $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
        $map = ['jpg'=>'image/jpeg','jpeg'=>'image/jpeg','png'=>'image/png','webp'=>'image/webp','gif'=>'image/gif'];
        $mime = $map[$ext] ?? 'application/octet-stream';
    }
    return $mime;
}

// create a JPEG thumbnail (keeps aspect, centers on canvas if needed)
function create_thumbnail($srcFile, $destFile, $maxW=260, $maxH=260, $quality=82) {
    $mime = detect_mime($srcFile, basename($srcFile));
    if (!in_array($mime, ['image/jpeg','image/png','image/webp','image/gif'])) return false;

    // load
    switch ($mime) {
        case 'image/jpeg': $im = @imagecreatefromjpeg($srcFile); break;
        case 'image/png' : $im = @imagecreatefrompng($srcFile); break;
        case 'image/webp': $im = function_exists('imagecreatefromwebp') ? @imagecreatefromwebp($srcFile) : false; break;
        case 'image/gif' : $im = @imagecreatefromgif($srcFile); break;
        default: $im = false;
    }
    if (!$im) return false;

    $w = imagesx($im); $h = imagesy($im);
    if ($w < 1 || $h < 1) { imagedestroy($im); return false; }

    // scale to fit max box
    $scale = min($maxW/$w, $maxH/$h, 1.0);
    $nw = (int)floor($w * $scale);
    $nh = (int)floor($h * $scale);

    // letterbox on white background
    $dst = imagecreatetruecolor($maxW, $maxH);
    $white = imagecolorallocate($dst, 255,255,255);
    imagefilledrectangle($dst, 0,0, $maxW, $maxH, $white);

    $dx = (int)floor(($maxW - $nw) / 2);
    $dy = (int)floor(($maxH - $nh) / 2);

    // for transparent png/gif -> white background above already applied
    imagecopyresampled($dst, $im, $dx,$dy, 0,0, $nw,$nh, $w,$h);

    // ensure folder exists
    @mkdir(dirname($destFile), 0755, true);
    $ok = imagejpeg($dst, $destFile, $quality);
    imagedestroy($im);
    imagedestroy($dst);
    return $ok;
}

// helper: thumb path from original basename (make .jpg)
function thumb_path($basename) {
    $name = pathinfo($basename, PATHINFO_FILENAME) . '.jpg';
    return [$GLOBALS['TH_DIR'] . '/' . $name, rtrim($GLOBALS['TH_URL'],'/').'/'.rawurlencode($name)];
}

/* ---------------- AJAX: Upload into Library (no DB) ---------------- */
if (isset($_GET['action']) && $_GET['action'] === 'upload_to_library') {
    header('Content-Type: application/json; charset=utf-8');

    if (empty($_FILES['files'])) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'msg' => 'No files uploaded']);
        exit;
    }

    $allowed = [
        'image/jpeg' => 'jpg',
        'image/png'  => 'png',
        'image/webp' => 'webp',
        'image/gif'  => 'gif'
    ];

    $results = [];
    $count = is_array($_FILES['files']['name']) ? count($_FILES['files']['name']) : 0;

    for ($i = 0; $i < $count; $i++) {
        $err = $_FILES['files']['error'][$i];
        if ($err !== UPLOAD_ERR_OK) {
            $results[] = ['ok'=>false, 'msg'=>"Upload error code $err"];
            continue;
        }

        $tmp  = $_FILES['files']['tmp_name'][$i];
        $orig = $_FILES['files']['name'][$i];
        $size = (int)($_FILES['files']['size'][$i] ?? 0);

        $mime = detect_mime($tmp, $orig);
        if (!isset($allowed[$mime])) {
            $results[] = ['ok'=>false,'msg'=>"Unsupported type: $mime",'name'=>$orig];
            continue;
        }
        $ext = $allowed[$mime];

        $base = pathinfo($orig, PATHINFO_FILENAME);
        $base = preg_replace('/[^A-Za-z0-9\-\._]+/u', '-', $base);
        $base = trim($base, '-_.');
        if ($base === '') $base = 'image';

        $target = $base . '.' . $ext;
        $n=1;
        while (file_exists($LIB_DIR . '/' . $target)) {
            $target = $base . '-' . $n++ . '.' . $ext;
        }

        if (!@move_uploaded_file($tmp, $LIB_DIR . '/' . $target)) {
            $results[] = ['ok'=>false,'msg'=>'Failed to save','name'=>$orig];
            continue;
        }

        // create thumbnail immediately
        list($thAbs, $thUrl) = thumb_path($target);
        @create_thumbnail($LIB_DIR.'/'.$target, $thAbs, 260, 260, 82);

        $url   = rtrim($LIB_URL, "/") . "/" . rawurlencode($target);
        $title = preg_replace('/[-_]+/',' ', pathinfo($target, PATHINFO_FILENAME));

        $results[] = [
            'ok'    => true,
            'url'   => $url,
            'thumb' => (is_file($thAbs) ? $thUrl : null),
            'name'  => $target,
            'title' => $title,
            'size'  => $size,
            'mime'  => $mime
        ];
    }

    echo json_encode(['ok'=>true,'items'=>$results]);
    exit;
}

/* ---------------- AJAX: List library (JSON, thumbs only) ---------------- */
if (isset($_GET['action']) && $_GET['action'] === 'list_library') {
    header('Content-Type: application/json; charset=utf-8');

    $items = [];
    $files = glob($LIB_DIR . "/*.{jpg,jpeg,png,webp,gif,JPG,JPEG,PNG,WEBP,GIF}", GLOB_BRACE) ?: [];
    foreach ($files as $abs) {
        $base = basename($abs);
        $url  = rtrim($LIB_URL, "/") . "/" . rawurlencode($base);
        $title= preg_replace('/[-_]+/',' ', pathinfo($base, PATHINFO_FILENAME));
        list($thAbs, $thUrl) = thumb_path($base);
        $items[] = [
            'url'   => $url,
            'thumb' => is_file($thAbs) ? $thUrl : null,
            'name'  => $base,
            'title' => $title,
            'size'  => @filesize($abs) ?: 0
        ];
    }
    // sort by name (case-insensitive)
    usort($items, fn($a,$b)=> strcasecmp($a['name'],$b['name']));

    echo json_encode(['ok'=>true, 'items'=>$items]);
    exit;
}

/* ---------------- AJAX: Build thumbnails for all ---------------- */
if (isset($_GET['action']) && $_GET['action'] === 'build_thumbs') {
    header('Content-Type: application/json; charset=utf-8');
    $built = 0; $failed = 0;
    $files = glob($LIB_DIR . "/*.{jpg,jpeg,png,webp,gif,JPG,JPEG,PNG,WEBP,GIF}", GLOB_BRACE) ?: [];
    foreach ($files as $abs) {
        $base = basename($abs);
        list($thAbs,) = thumb_path($base);
        if (is_file($thAbs)) continue; // already exists
        if (create_thumbnail($abs, $thAbs, 260, 260, 82)) $built++; else $failed++;
    }
    echo json_encode(['ok'=>true, 'built'=>$built, 'failed'=>$failed]);
    exit;
}

/* ---------------- Scan library for counts ONLY (no heavy images at render) ---------------- */
$libraryCount = 0;
if (is_dir($LIB_DIR)) {
    $files = glob($LIB_DIR . "/*.{jpg,jpeg,png,webp,gif,JPG,JPEG,PNG,WEBP,GIF}", GLOB_BRACE) ?: [];
    $libraryCount = count($files);
}

/* ---------------- EDIT MODE ---------------- */
$product = [];
$productPrices = [];
$currentCat = 0;
$currentSub = 0;

if (isset($_GET['id']) && ctype_digit($_GET['id'])) {
    $stmt = $conn->prepare("SELECT * FROM shop_products WHERE id=?");
    $stmt->bind_param("i", $_GET['id']);
    $stmt->execute();
    $product = $stmt->get_result()->fetch_assoc() ?: [];
    $currentCat = isset($product['category']) ? (int)$product['category'] : 0;
    $currentSub = isset($product['subcategory']) ? (int)$product['subcategory'] : 0;

    $pp = $conn->prepare("SELECT weight,unit,price,barcode FROM shop_product_prices WHERE product_id=?");
    $pp->bind_param("i", $_GET['id']);
    $pp->execute();
    $productPrices = $pp->get_result()->fetch_all(MYSQLI_ASSOC);
}

/* ---------------- MASTER LISTS ---------------- */
$makes  = $conn->query("SELECT id,category_name FROM shop_categories ORDER BY category_name");
$models = $conn->query("SELECT id,sub_category_name FROM shop_sub_categories ORDER BY sub_category_name");

/* ---------------- Helpers for gallery JSON (strings or objects) ---------------- */
function normalize_gallery($rawJson) {
    if (!$rawJson) return [];
    $arr = json_decode($rawJson, true);
    if (!is_array($arr)) return [];
    $out = [];
    foreach ($arr as $item) {
        if (is_string($item)) {
            $out[] = ['url'=>$item, 'title'=>''];
        } elseif (is_array($item) && isset($item['url'])) {
            $out[] = ['url'=>$item['url'], 'title'=>$item['title'] ?? ''];
        }
    }
    return $out;
}
$existingGallery = normalize_gallery($product['gallery'] ?? '');
$existingMainUrl = $product['images'] ?? '';
$existingMainTitle = $product['main_img_title'] ?? '';

?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
  <meta charset="utf-8">
  <title><?= $product ? 'עריכת מוצר' : 'הוספת מוצר' ?></title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link rel="stylesheet" href="app/css/font-awesome.css">
  <link rel="stylesheet" href="app/css/styles.css">

  <style>
  html, body, #wrapper-dashboard { overflow-x: hidden; }

  .media-source-wrap{ display:flex; gap:1rem; align-items:center; margin:.5rem 0 1rem; }
  .media-source-wrap label {margin-bottom:0;}

  #mainLibraryBlock, #mainUploadBlock{ display:block; width:100%; max-width:100%; }

  #mainLibraryBlock .row, #mainUploadBlock .row{ margin-right:0 !important; margin-left:0 !important; }

  .library-grid {display:flex; flex-wrap:wrap; gap:10px;}
  .library-item {border:1px solid #ddd; border-radius:6px; padding:6px; width:130px; text-align:center; position:relative; background:#fff;}
  .library-item img {max-width:100%; height:90px; object-fit:cover; border-radius:4px;}
  .library-item input[type="checkbox"] {position:absolute; top:6px; left:6px; transform:scale(1.2); z-index:10}
  .library-item .item-name {font-size:.75rem; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; direction:ltr;}
  .library-item .title-input {width:100%; margin-top:6px;}
  .thumb {position:relative; border:1px dashed #ccc; border-radius:6px; padding:4px; display:inline-block;}
  .thumb .remove {position:absolute; top:-8px; left:-8px; background:#e74c3c; color:#fff; border-radius:50%; width:20px; height:20px; line-height:18px; text-align:center; cursor:pointer;}
  .thumb .title-under {font-size:.8rem; margin-top:4px;}
  .thumb .drag-handle {position:absolute; bottom:-8px; left:-8px; background:#6b7280; color:#fff; border-radius:50%; width:20px; height:20px; line-height:18px; text-align:center; cursor:grab; font-size:12px;}
  .modal .modal-box {background:#fff; border-radius:8px; padding:16px;}
  .modal .modal-title {margin-bottom:.5rem;}
  .badge-note {font-size:.8rem; color:#666;}
  #storyModal {z-index: 30000 !important}

  .wizard-steps { margin: 8px 0 0; }
  .wizard-steps .nav-link { font-weight:600; }
  .wizard-pane { padding:16px; border:1px solid #eef2f7; border-top:none; border-radius:0 0 .5rem .5rem; background:#fff; }
  .wizard-actions { display:flex; gap:8px; justify-content:space-between; margin-top:10px; }
  .wizard-actions .btn { min-width:110px; }
  .wizard-progress { height:6px; background:#eef2f7; border-radius:999px; overflow:hidden; margin-top:10px; }
  .wizard-progress .bar { height:100%; width:0; background:#3b82f6; transition:width .25s ease; }

  .field-hint { font-size:.85rem; color:#64748b; margin-top:6px; }
  .inline-help { font-size: .85rem; color: #6b7280; display:block; margin-top:4px; }
  .required-asterisk::after { content:" *"; color:#e11d48; }

  .sub-filter { display:flex; gap:8px; align-items:center; }
  .sub-filter input { max-width:220px; }

  .media-tabs { margin-top:16px; }
  .media-tabs .nav-link { font-weight:600; }
  .media-pane{
    border:1px solid #e5e7eb; border-radius:12px; background:#fff; display:flex; flex-direction:column; min-height:0; margin-top:8px;
  }
  .pane-header{
    padding:10px 12px; border-bottom:1px solid #eef2f7; display:flex; align-items:center; justify-content:space-between; gap:.75rem;
  }
  .pane-header h4{ margin:0; font-size:1rem; font-weight:600; }
  .pane-hint{ margin:0; font-size:.85rem; color:#64748b; }
  .pane-body{ padding:12px; overflow:auto; min-height:0; }
  #mainImgPreview{
    min-height:150px; border:1px solid #e5e7eb; border-radius:10px; padding:8px; background:#fff;
  }
  #mainImgPreview img { max-width: 500px; height:auto; } /* limit main preview */

  .save-bar{
    position:sticky; bottom:0; background:#fff; border-top:1px solid #e5e7eb; padding:10px 12px; display:flex; align-items:center; justify-content:space-between; gap:10px; z-index:5; margin-top:18px; border-radius:10px 10px 0 0;
  }
  .save-bar .meta{ font-size:.9rem; color:#475569; }
  .save-bar .actions{ display:flex; gap:8px; }

  .is-invalid { border-color:#ef4444 !important; }
  .count-badge{ display:inline-block; padding:2px 8px; background:#eef2ff; color:#3730a3; border-radius:999px; font-size:.8rem; }

  /* simple spinner */
  .spinner {display:inline-block; width:18px; height:18px; border:2px solid #e5e7eb; border-top-color:#3b82f6; border-radius:50%; animation:spin 0.8s linear infinite; vertical-align:middle}
  @keyframes spin {to{transform:rotate(360deg)}}
  </style>
</head>


<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/cropperjs@1.6.2/dist/cropper.min.css">
<script src="https://cdn.jsdelivr.net/npm/cropperjs@1.6.2/dist/cropper.min.js"></script>
<script src="/app/js/image-editor.js"></script>



<body class="body">

<?php include './includes/cms/aside.php'; ?>

<div id="wrapper-dashboard">
  <?php include './includes/cms/header.php'; ?>

  <div class="container my-4">

    <h1><?= $product ? 'עריכת מוצר' : 'הוספת מוצר' ?></h1>

    <form id="productForm" action="save_product.php" method="POST" enctype="multipart/form-data" method="POST" enctype="multipart/form-data">

      <!-- ===== Wizard Progress ===== -->
      <div class="wizard-progress" aria-hidden="true">
        <div class="bar" id="wizardBar" style="width:0%"></div>
      </div>

      <!-- ===== LEFT: Wizard (now full width) ===== -->
      <div class="row">
        <div class="col-lg-12">
          <div class="tfcl-add-listing car-details">

            <!-- Wizard Tabs -->
            <ul class="nav nav-tabs wizard-steps" id="wizardTabs" role="tablist">
              <li class="nav-item">
                <a class="nav-link active" data-toggle="tab" href="#tabCats" role="tab" data-step="1">1. קטגוריות</a>
              </li>
              <li class="nav-item">
                <a class="nav-link" data-toggle="tab" href="#tabContent" role="tab" data-step="2">2. תוכן & תרגום</a>
              </li>
              <li class="nav-item">
                <a class="nav-link" data-toggle="tab" href="#tabPrices" role="tab" data-step="3">3. יחידות & מחירים</a>
              </li>
            </ul>

            <div class="tab-content wizard-pane">

              <!-- Step 1: Categories -->
              <div class="tab-pane fade show active" id="tabCats" role="tabpanel" aria-labelledby="cats-tab">
                <div class="form-row">
                  <div class="form-group col-md-6">
                    <label class="required-asterisk">קטגוריה</label>
                    <select name="main_id" id="main_id" class="form-control" required aria-describedby="catHelp">
                      <option value="">– בחר –</option>
                      <?php while ($m = $makes->fetch_assoc()): ?>
                        <?php $mid = (int)$m['id']; ?>
                        <option value="<?= $mid ?>" <?= ($currentCat===$mid?'selected':'') ?>>
                          <?= htmlspecialchars($m['category_name']) ?>
                        </option>
                      <?php endwhile; ?>
                      <option value="__new">+ קטגוריה חדשה…</option>
                    </select>
                    <span id="catHelp" class="inline-help">ניתן להוסיף קטגוריה חדשה מהתפריט.</span>
                  </div>

                  <div class="form-group col-md-6">
                    <label >קטגוריה משנית <span id="subCount" class="count-badge" style="display:none"></span></label>
                    <div class="sub-filter">
                      <select name="model_id" id="model_id" class="form-control" aria-describedby="subHelp" <?= $currentCat ? '' : 'disabled' ?>>
                        <option value="">– בחר –</option>
                        <?php while ($md = $models->fetch_assoc()): ?>
                          <?php $sdid=(int)$md['id']; ?>
                          <option value="<?= $sdid ?>" <?= ($currentSub===$sdid?'selected':'') ?>>
                            <?= htmlspecialchars($md['sub_category_name']) ?>
                          </option>
                        <?php endwhile; ?>
                        <option value="__new">+ קטגוריה משנית חדשה…</option>
                      </select>
                      <input type="text" id="subQuickFilter" class="form-control" placeholder="חיפוש מהיר" <?= $currentCat ? '' : 'disabled' ?>>
                      <button type="button" class="btn btn-light" id="clearSubFilter" title="נקה חיפוש">✕</button>
                    </div>
                    <small id="subHelp" class="inline-help">בחר קטגוריה ראשית כדי לצמצם אפשרויות.</small>
                  </div>
                </div>

                <div class="wizard-actions">
                  <div></div>
                  <button type="button" class="btn btn-secondary" id="toContent" disabled>לתוכן →</button>
                </div>
              </div>

              <!-- Step 2: Content -->
              <div class="tab-pane fade" id="tabContent" role="tabpanel" aria-labelledby="content-tab">
                <ul class="nav nav-tabs" id="langTabs" role="tablist">
                  <li class="nav-item">
                    <a class="nav-link active" data-toggle="tab" href="#heTab">עברית</a>
                  </li>
                  <li class="nav-item">
                    <a class="nav-link" data-toggle="tab" href="#arTab">العربية</a>
                  </li>
                </ul>
                <div class="tab-content border p-3">
                  <div class="tab-pane fade show active" id="heTab">
                    <div class="form-group">
                      <label>כותרת (עברית)</label>
                      <input type="text" name="title_he" class="form-control"
                             value="<?= htmlspecialchars($product['title_he'] ?? '') ?>">
                    </div>
                    <div class="form-group">
                      <label>תיאור (עברית)</label>
                      <textarea name="description_he" rows="3"
                                class="form-control"><?= htmlspecialchars($product['description_he'] ?? '') ?></textarea>
                    </div>
                    <div class="form-group">
                      <label>מרכיבים (עברית)</label>
                      <textarea name="ingredients_he" rows="2"
                                class="form-control"><?= htmlspecialchars($product['ingredients_he'] ?? '') ?></textarea>
                    </div>
                  </div>
                  <div class="tab-pane fade" id="arTab">
                    <div class="form-group">
                      <label>العنوان</label>
                      <input type="text" name="title_ar" class="form-control"
                             value="<?= htmlspecialchars($product['title_ar'] ?? '') ?>">
                    </div>
                    <div class="form-group">
                      <label>الوصف</label>
                      <textarea name="description_ar" rows="3"
                                class="form-control"><?= htmlspecialchars($product['description_ar'] ?? '') ?></textarea>
                    </div>
                    <div class="form-group">
                      <label>المكونات</label>
                      <textarea name="ingredients_ar" rows="2"
                                class="form-control"><?= htmlspecialchars($product['ingredients_ar'] ?? '') ?></textarea>
                    </div>
                  </div>
                </div>

                <div class="wizard-actions">
                  <button type="button" class="btn btn-light" id="backToCats">← חזרה</button>
                  <button type="button" class="btn btn-secondary" id="toPrices">לתמחור →</button>
                </div>
              </div>

              <!-- Step 3: Pricing -->
              <div class="tab-pane fade" id="tabPrices" role="tabpanel" aria-labelledby="prices-tab">
                <h4 class="mt-2">יחידות ומשקל</h4>
                <p class="field-hint">הוסף כמה שורות שצריך. ניתן לגרור לשינוי סדר ולהסיר בעזרת ⨯.</p>
                <table class="table table-sm" id="priceTable">
                  <thead class="thead-light">
                    <tr><th>כמות</th><th>יחידה</th><th>מחיר</th><th>ברקוד</th><th></th></tr>
                  </thead>
                  <tbody>
                  <?php if ($productPrices): foreach ($productPrices as $pp): ?>
                    <tr>
                      <td><input type="number" step="any" name="weights[]"
                                 value="<?= htmlspecialchars($pp['weight']) ?>" class="form-control"></td>
                      <td>
                        <select name="units[]" class="form-control">
                          <?php foreach (['kg','ton','g','unit'] as $u): ?>
                            <option value="<?= $u ?>" <?= ($u==$pp['unit']?'selected':'')?>><?= strtoupper($u) ?></option>
                          <?php endforeach; ?>
                        </select>
                      </td>
                      <td><input type="number" step="0.01" name="prices[]"
                                 value="<?= htmlspecialchars($pp['price']) ?>" class="form-control"></td>
                      <td><input type="text" name="barcodes[]"
                                 value="<?= htmlspecialchars($pp['barcode']) ?>" class="form-control" placeholder="1234-5678"></td>
                      <td><button type="button" class="btn btn-danger btn-sm remove-row" aria-label="מחק">×</button></td>
                    </tr>
                  <?php endforeach; endif; ?>
                  </tbody>
                </table>
                <div class="d-flex align-items-center gap-2">
                  <button type="button" id="addPriceRow" class="btn btn-secondary btn-sm">הוסף מחיר</button>
                  <span class="inline-help" id="priceIssues" style="display:none"></span>
                </div>

                <div class="wizard-actions">
                  <button type="button" class="btn btn-light" id="backToContent">← חזרה</button>
                  <a href="#mediaSection" class="btn btn-outline-primary">למדיה ↓</a>
                </div>
              </div>

            </div><!-- /.tab-content -->

          </div>
        </div>
      </div>

      <!-- ===== MEDIA: now with tabs (Main / Gallery) ===== -->
      <div class="row" id="mediaSection">
        <div class="col-lg-12">
          <div class="tfcl-add-listing car-details">
            <h3 tabindex="-1">מדיה <span class="badge-note">(תמונה ראשית + גלריה, ללא DB)</span></h3>

            <!-- Media Tabs -->
            <div class="media-tabs">
              <ul class="nav nav-tabs" id="mediaTabs" role="tablist">
                <li class="nav-item">
                  <a class="nav-link active" data-toggle="tab" href="#tabMainImg" role="tab">תמונה ראשית</a>
                </li>
                <li class="nav-item">
                  <a class="nav-link" data-toggle="tab" href="#tabGallery" role="tab">גלריה</a>
                </li>
              </ul>

              <div class="tab-content">

                <!-- MAIN IMAGE TAB -->
                <div class="tab-pane fade show active" id="tabMainImg" role="tabpanel">
                  <section class="media-pane" id="paneMain">
                    <div class="pane-header">
                      <h4>תמונה ראשית</h4>
                      <p class="pane-hint">בחר מקור, העלה/חתוך (1×1), ותן כותרת.</p>
                    </div>
                    <div class="pane-body">
                      <div class="media-source-wrap">
                        <strong>מקור:</strong>
                        <label class="ml-2"><input type="radio" name="main_img_source" value="upload" checked> העלאה</label>
                        <label class="ml-2"><input type="radio" name="main_img_source" value="library"> ספרייה</label>
                      </div>

                      <!-- Upload block -->
                      <div id="mainUploadBlock" style="display:block">
                        <div class="d-flex flex-row flex-wrap mb-3">
                          <div class="pr-3" style="min-width:240px;">
                            <label>תמונה ראשית (יחס 1×1)</label>
                            <input type="hidden" id="mainImgUrl" name="main_img_url" value="<?= htmlspecialchars($existingMainUrl) ?>" >
<input type="hidden" name="image_full" id="image_full">
<input type="hidden" name="image_226"  id="image_226">
<input type="hidden" name="image_tiny" id="image_tiny">">
                            <input type="hidden" name="main_img_cropped" id="mainImgCropped">
                            <label class="btn btn-outline-primary btn-block">
                              העלה קובץ
                              <input type="file" id="mainImgFile" hidden accept="image/*" onchange="ImageEditor.upload(this, \'#img-previews\')">
                            </label>
                            <input type="text" class="form-control mt-2"
                                  name="main_img_title" id="mainImgTitleInput"
                                  placeholder="כותרת לתמונה" value="<?= htmlspecialchars($existingMainTitle) ?>">
                          </div>
                          <div class="flex-grow-1">
                            <div class="row">
                                <div class="col-lg-6"> <div id="mainImgPreview" class="d-flex align-items-center justify-content-center">
                              <?php if(!empty($existingMainUrl)): ?>
                                <img src="<?= htmlspecialchars($existingMainUrl) ?>" alt="" />
                              <?php endif; ?>
                            </div>
                          </div>
                                <div class="col-lg-6">
<div class="mt-3">
  <div class="mb-2">חיתוך INLINE (1×1)</div>
  <div class="ratio ratio-1x1" style="max-width:420px">
    <img id="inline-crop-img" src="" style="max-width:100%">
  </div>
  <div class="mt-2 d-flex gap-2">
    <button type="button" class="btn btn-outline-primary" onclick="ImageEditor.applyCrop('#img-previews')">יישם חיתוך</button>
    <button type="button" class="btn btn-outline-secondary" onclick="ImageEditor.padSquare('#img-previews')">הוסף שטח לבן ל־1×1</button>
  </div>
  <div id="img-previews" class="mt-2"></div>
</div>

                                </div>

                            </div>
                           

                          </div>
                        </div>
                      </div>

                      <!-- Library block -->
                      <div id="mainLibraryBlock" style="display:none">
                        <div class="mb-2 d-flex align-items-center justify-content-between">
                          <div class="d-flex align-items-center gap-2">
                            <button type="button" class="btn btn-outline-secondary" id="openMainLibrary">בחר מתוך הספרייה</button>
                            <span class="inline-help">בחר בדיוק תמונה אחת.</span>
                          </div>
                          <div class="d-flex align-items-center gap-2">
                            <button type="button" class="btn btn-light btn-sm" id="buildThumbsBtn">בנה תמונות ממוזערות</button>
                            <span class="text-muted">בספרייה: <?= (int)$libraryCount ?> קבצים</span>
                          </div>
                        </div>
                        <div id="mainChosenFromLib" class="d-flex align-items-start"></div>
                        <input type="hidden" id="mainImgUrlLib" name="main_img_url_lib" value="">
                      </div>

                      <!-- crop-dialog -->
                      <div id="cropModal" class="modal" style="display:none">
                        <div class="modal-box" style="max-width:600px">
                          <h3 class="modal-title">חתוך תמונה</h3>
                          <div class="modal-body"><img id="cropperImg" style="max-width:100%"></div>
                          <div class="modal-buttons">
                            <button type="button" class="sc-button btn-cancel" id="cropCancel">ביטול</button>
                            <button type="button" class="sc-button btn-ok" id="cropApply">החל</button>
                          </div>
                        </div>
                      </div>
                    </div>
                  </section>
                </div>

                <!-- GALLERY TAB -->
                <div class="tab-pane fade" id="tabGallery" role="tabpanel">
                  <section class="media-pane" id="paneGallery">
                    <div class="pane-header">
                      <h4>גלריה</h4>
                      <p class="pane-hint">עד 10 תמונות בסך הכל. גרור לשינוי סדר.</p>
                    </div>
                    <div class="pane-body">
                      <div class="gallery-box">
                        <?php if (!empty($existingGallery)): ?>
                          <h5>תמונות קיימות</h5>
                          <div class="gallery-wrap d-flex flex-wrap" id="oldGallery">
                            <?php foreach($existingGallery as $img): ?>
                              <label class="thumb m-1">
                                <span class="remove" title="מחק">×</span>
                                <span class="drag-handle" title="גרור">≡</span>
                                <img src="<?= htmlspecialchars($img['url']) ?>" style="height:80px" alt="">
                                <input type="hidden" value='<?= htmlspecialchars($img['url'], ENT_QUOTES) ?>'>
                                <div class="title-under">
                                  <input type="text" class="form-control form-control-sm existing-gallery-title"
                                         value="<?= htmlspecialchars($img['title']) ?>" placeholder="כותרת לתמונה">
                                </div>
                              </label>
                            <?php endforeach; ?>
                          </div>
                        <?php endif; ?>

                        <div class="d-flex align-items-center justify-content-between mt-2">
                          <h5 class="m-0">העלאת תמונות חדשות</h5>
                          <div>
                            <span class="count-badge" id="galleryCountBadge" style="display:none"></span>
                            <button type="button" class="btn btn-light btn-sm" id="clearNewGallery" title="נקה תמונות חדשות">נקה חדשות</button>
                          </div>
                        </div>

                        <input type="file" multiple accept="image/*" id="galleryInput" name="images[]">
                        <small class="d-block text-muted mb-2">ניתן לבחור כמה קבצים יחד.</small>
                        <div id="newGallery" class="d-flex flex-wrap"></div>

                        <div class="mt-2 d-flex align-items-center gap-2">
                          <button type="button" class="btn btn-outline-secondary btn-sm" id="openGalleryLibrary">הוסף מהספרייה</button>
                          <small class="text-muted">תמונות נבחרות יתווספו בתחתית. מקסימום 10 בסך הכל.</small>
                        </div>
                      </div>
                    </div>
                  </section>
                </div>

              </div><!-- /.tab-content -->
            </div><!-- /.media-tabs -->

            <!-- Sticky Save Bar -->
            <div class="save-bar">
              <div class="meta">
                <span id="summaryCat">קטגוריה: —</span> ·
                <span id="summarySub">משנית: —</span> ·
                <span id="summaryPrices">שורות מחירים: 0</span> ·
                <span id="summaryGallery">תמונות בגלריה: 0</span>
              </div>
              <div class="actions">
                <button class="pre-btn">שמור</button>
                <button name="save_preview" class="second-btn">שמור + תצוגה</button>
              </div>
            </div>

          </div>
        </div>
      </div>

      <!-- hidden -->
      <input type="hidden" name="product_id"       value="<?= $product['id'] ?? '' ?>">
      <input type="hidden" name="old_images"       value="<?= htmlspecialchars($existingMainUrl) ?>">
      <input type="hidden" name="old_gallery_json" value="<?= htmlspecialchars($product['gallery'] ?? '',ENT_QUOTES) ?>">
      <input type="hidden" name="gallery_sort_json" id="gallerySort">
      <input type="hidden" name="gallery_titles_json" id="galleryTitlesJson">

    </form>
  </div>
</div>

<!-- ---------- MODALS ---------- -->
<div id="storyModal" class="modal"><div class="modal-box"></div></div>

<!-- Library modal – now empty grid; will be filled lazily -->
<div id="libraryModal" class="modal" style="display:none">
  <div class="modal-box" style="max-width:980px">
    <h3 class="modal-title d-flex align-items-center justify-content-between">
      <span>בחר / העלה תמונות לספרייה</span>
      <span id="libTopInfo" class="text-muted" style="font-size:.9rem"></span>
    </h3>

    <div id="libUploader" class="library-uploader-modern">
      <div class="lib-dropzone">
        <p>גרור תמונות לכאן או</p>
        <label class="btn-upload">
          בחר קבצים
          <input type="file" id="libFiles" hidden multiple accept="image/*">
        </label>
        <button type="button" id="libUploadBtn" class="btn-confirm">העלה לספרייה</button>
        <small class="lib-note">סוגים נתמכים: JPG / PNG / WebP / GIF</small>
        <div id="libUploadStatus" class="status"></div>
      </div>
    </div>

    <div class="modal-body">
      <div class="d-flex align-items-center justify-content-between mb-2">
        <div>
          <span id="libSelectedCount" class="count-badge" style="display:none"></span>
          <span id="libLoading" style="display:none"><span class="spinner"></span> טוען ספרייה…</span>
        </div>
        <div class="d-flex align-items-center gap-2">
          <button type="button" id="libBuildThumbs" class="btn btn-light btn-sm">בנה תמונות ממוזערות</button>
          <button type="button" id="libSelectNone" class="btn btn-light btn-sm">נקה בחירה</button>
        </div>
      </div>

      <div class="library-grid" id="libraryGrid">
        <!-- populated lazily with thumbs -->
      </div>
    </div>
    <div class="modal-buttons">
      <button type="button" class="sc-button btn-cancel" id="libCancel">ביטול</button>
      <button type="button" class="sc-button btn-ok" id="libApply" disabled>הוסף</button>
    </div>
  </div>
</div>

<?php include './includes/cms/modales.php'; ?>
<script src="app/js/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/Sortable/1.15.2/Sortable.min.js" crossorigin="anonymous"></script>
<script src="app/js/modalUtil.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/cropperjs/1.5.13/cropper.min.js"></script>

<script>
/* ==========================================================
   CONSTANTS
   ========================================================== */
const GALLERY_MAX = 10;

/* ==========================================================
   CATEGORY + SUB-CATEGORY
   ========================================================== */
let CATS = [], SUBS = {};

function renderCats(selectedId = 0){
  const $c = $('#main_id');
  $c.find('option:not([value=""],[value="__new"])').remove();
  CATS.forEach(c =>
    $('<option>',{value:c.id,text:c.category_name})
      .insertBefore($c.find('[value="__new"]')));
  if (selectedId) $c.val(String(selectedId));
}

function renderSubs(catId, selectedId = 0){
  const $s = $('#model_id');
  const quick = ($('#subQuickFilter').val()||'').toLowerCase().trim();
  $s.find('option:not([value=""],[value="__new"])').remove();

  const items = (SUBS[catId]||[]);
  let shown = 0;
  items.forEach(s => {
    const txt = String(s.sub_category_name||'');
    if (!quick || txt.toLowerCase().includes(quick)) {
      $('<option>',{value:s.id,text:txt})
        .insertBefore($s.find('[value="__new"]'));
      shown++;
    }
  });
  if (selectedId) $s.val(String(selectedId));
  $('#subCount').text(`${shown} תתי-קטגוריות`).toggle(shown>0);
}

function fetchCatLists(cb){
  $.getJSON('shop_ajax_category.php',{action:'make_model_lists'})
   .done(res => { CATS=res.makes||[]; SUBS=res.models||{}; cb&&cb(); })
   .fail(()=>modalUtil.alert('שגיאה בטעינת קטגוריות','error'));
}

$('#main_id').on('change',function(){
  const val = this.value;
  const isNew = (val==='__new');
  if (!isNew) {
    $('#model_id, #subQuickFilter, #toContent').prop('disabled', false);
    renderSubs(val);
  }
});

$('select#main_id, select#model_id').on('change',function(){
  if (this.value!=='__new') return;
  const isMain=this.id==='main_id';
  if (!isMain && !$('#main_id').val()) {
    modalUtil.alert('בחר קודם קטגוריה ראשית','error');
    $(this).val('');
    return;
  }
  modalUtil.open({
    title : isMain?'הוסף קטגוריה חדשה':'הוסף תת-קטגוריה חדשה',
    body  : '<input id="newOptionValue" class="form-control" style="width:100%" autofocus>',
    buttons:[
      {text:'ביטול',class:'btn-cancel'},
      {text:'שמור', class:'btn-ok',callback:()=>saveNewOption(isMain)}
    ]
  });
  $(this).val('');
});

function saveNewOption(isMain){
  const name=$('#newOptionValue').val().trim();
  if(!name) return modalUtil.alert('נא למלא שם','error');
  const data=isMain
        ? {action:'add_category',name}
        : {action:'add_sub',main_id:$('#main_id').val(),name};
  $.post('shop_ajax_category.php',data,res=>{
     try{res=typeof res==='string'?JSON.parse(res):res;}catch(e){res={ok:false};}
     if(!res.ok) return modalUtil.alert(res.msg||'שגיאה','error');
     fetchCatLists(()=>{
        if(isMain){renderCats(res.id);renderSubs(res.id);}
        else      {renderSubs($('#main_id').val(),res.id);}
        modalUtil.alert('נשמר בהצלחה!','success')
                 .then(()=>modalUtil.close());
     });
  }).fail(()=>modalUtil.alert('שגיאת תקשורת','error'));
}

/* Quick filter for subcategories */
$('#subQuickFilter').on('input', function(){
  const catId = $('#main_id').val();
  if (catId) renderSubs(catId);
});
$('#clearSubFilter').on('click', function(){
  $('#subQuickFilter').val('');
  const catId = $('#main_id').val();
  if (catId) renderSubs(catId);
});

/* Enable "Next" only when cat & sub chosen */
function validateStep1(){
  const ok = !!$('#main_id').val() && !!$('#model_id').val();
  $('#toContent').prop('disabled', !ok);
  $('#main_id').toggleClass('is-invalid', !$('#main_id').val());
  $('#model_id').toggleClass('is-invalid', !!$('#main_id').val() && !$('#model_id').val());
  updateSummary();
}
$('#main_id, #model_id').on('change', validateStep1);

/* ==========================================================
   LEFT WIZARD: Next/Back helpers (tabs) + persistence
   ========================================================== */
const WIZARD_KEY = 'ap_wizard_tab';
const MEDIA_KEY  = 'ap_media_tab';

function showTab(selector, saveKey=true){
  $('a[href="'+selector+'"]').tab('show');
  if (saveKey) localStorage.setItem(WIZARD_KEY, selector);
  updateWizardProgress();
}
$('#toContent').on('click', ()=>showTab('#tabContent'));
$('#toPrices').on('click', ()=>showTab('#tabPrices'));
$('#backToCats').on('click', ()=>showTab('#tabCats'));
$('#backToContent').on('click', ()=>showTab('#tabContent'));

$('#wizardTabs a[data-toggle="tab"]').on('shown.bs.tab', updateWizardProgress);
function updateWizardProgress(){
  const $active = $('#wizardTabs .nav-link.active');
  const step = parseInt($active.data('step')||1,10);
  const percent = (step-1) / 2 * 100; // 3 steps -> 0%, 50%, 100%
  $('#wizardBar').css('width', percent+'%');
}

/* Restore last tabs on load */
function restoreTabs(){
  const wiz = localStorage.getItem(WIZARD_KEY);
  if (wiz && $(wiz).length) $('a[href="'+wiz+'"]').tab('show');
  updateWizardProgress();

  const med = localStorage.getItem(MEDIA_KEY);
  if (med && $(med).length) $('a[href="'+med+'"]').tab('show');
}
$('#mediaTabs a[data-toggle="tab"]').on('shown.bs.tab', function(e){
  localStorage.setItem(MEDIA_KEY, $(e.target).attr('href'));
});

/* ==========================================================
   MEDIA: Main image (Upload vs Library) + Cropper
   ========================================================== */
function toggleMainSource() {
  const lib = $('input[name="main_img_source"]:checked').val()==='library';
  $('#mainUploadBlock').css('display', lib ? 'none'  : 'block');
  $('#mainLibraryBlock').css('display', lib ? 'block' : 'none');

  // Reset horizontal scroll & focus safe element
  requestAnimationFrame(()=>{
    document.documentElement.scrollLeft = 0;
    document.body.scrollLeft = 0;
    const safe = document.querySelector('h3');
    if (safe) safe.tabIndex = -1, safe.focus();
  });
}
$('input[name="main_img_source"]').on('change', toggleMainSource);
toggleMainSource();

let cropper=null;
$('#mainImgFile').on('change',function(){
  const f=this.files[0];
  if(!f) return;
  const r=new FileReader();
  r.onload=e=>{
     $('#cropperImg').attr('src',e.target.result);
     $('#cropModal').fadeIn(120);
     cropper && cropper.destroy();
     cropper = new Cropper(document.getElementById('cropperImg'),{
        aspectRatio:1,
        viewMode:1,
        dragMode:'move',
        background:false
     });
  };
  r.readAsDataURL(f);
});
$('#cropCancel').on('click',()=>$('#cropModal').fadeOut(120));
$('#cropApply').on('click',function(){
   const canvas=cropper.getCroppedCanvas({width:800,height:800});
   const dataURL=canvas.toDataURL('image/jpeg',0.9);
   $('#mainImgPreview').html('<img src="'+dataURL+'" alt="">');
   $('#mainImgCropped').val(dataURL);
   $('#mainImgUrl').val('');
   $('#cropModal').fadeOut(120);
});

/* ==========================================================
   MEDIA LIBRARY modal + uploader (Lazy load)
   ========================================================== */
let LIB_MODE = null;
let LIB_LOADED = false;
$('#openMainLibrary').on('click', function(){
  LIB_MODE = 'main';
  openLibraryModal(1);
});
$('#openGalleryLibrary').on('click', function(){
  LIB_MODE = 'gallery';
  openLibraryModal(0);
});

// extra buttons outside modal
$('#buildThumbsBtn').on('click', buildThumbsAll);

function openLibraryModal(maxSelect=0){
  $('#libraryModal .lib-pick').prop('checked', false);
  $('#libSelectedCount').hide().text('');
  $('#libApply').prop('disabled', LIB_MODE==='main'); // need exactly 1 for main
  $('#libraryModal').fadeIn(120).data('max', maxSelect);
  loadLibraryGrid(); // lazy load every open (fast because thumbs)
}
$('#libCancel').on('click', ()=> $('#libraryModal').fadeOut(120));

async function loadLibraryGrid(){
  try{
    $('#libraryGrid').empty();
    $('#libLoading').show();
    $('#libTopInfo').text('');
    const resp = await fetch('add-product.php?action=list_library');
    const data = await resp.json();
    $('#libLoading').hide();
    if (!data.ok) throw new Error('failed');
    const items = data.items || [];
    $('#libTopInfo').text(`סה"כ ${items.length} קבצים`);

    if (!items.length) {
      $('#libraryGrid').html('<p class="text-muted m-2">אין תמונות בספרייה. העלה קבצים בעזרת האזור מעלה.</p>');
      return;
    }
    const frag = document.createDocumentFragment();
    items.forEach(it=>{
      const thumb = it.thumb || it.url; // fallback: original (rare)
      const el = document.createElement('label');
      el.className = 'library-item';
      el.innerHTML = `
        <input type="checkbox" class="lib-pick" data-url="${it.url}">
        <img src="${thumb}" alt="">
        <div class="item-name" title="${it.name}">${it.name}</div>
        <input type="text" class="form-control form-control-sm title-input" placeholder="כותרת" value="${it.title}">
      `;
      frag.appendChild(el);
    });
    $('#libraryGrid').append(frag);
    LIB_LOADED = true;
  }catch(e){
    console.error(e);
    $('#libLoading').hide();
    $('#libraryGrid').html('<p class="text-danger">שגיאה בטעינת הספרייה.</p>');
  }
}

/* Build thumbs (modal button and external button share the same handler) */
$('#libBuildThumbs').on('click', buildThumbsAll);
async function buildThumbsAll(){
  try{
    const btns = $('#libBuildThumbs, #buildThumbsBtn').prop('disabled', true);
    const oldTxt = btns.first().text();
    btns.text('בונה…');
    const resp = await fetch('add-product.php?action=build_thumbs');
    const data = await resp.json();
    btns.prop('disabled', false).text(oldTxt);
    if (!data.ok) throw new Error('fail');
    modalUtil.alert(`נבנו ${data.built} תמונות ממוזערות. נכשלו: ${data.failed||0}`, 'success');
    // reload library grid to use fresh thumbs
    if ($('#libraryModal').is(':visible')) loadLibraryGrid();
  }catch(e){
    console.error(e);
    modalUtil.alert('שגיאה בבניית תמונות ממוזערות', 'error');
    $('#libBuildThumbs, #buildThumbsBtn').prop('disabled', false).text('בנה תמונות ממוזערות');
  }
}

/* Selection counter & rules */
$(document).on('change', '#libraryGrid .lib-pick', function(){
  const count = $('#libraryGrid .lib-pick:checked').length;
  if (count>0) { $('#libSelectedCount').text(`${count} נבחרו`).show(); } else { $('#libSelectedCount').hide(); }
  if (LIB_MODE==='main') {
    $('#libApply').prop('disabled', count!==1);
  } else {
    $('#libApply').prop('disabled', count===0);
  }
});
$('#libSelectNone').on('click', ()=>{
  $('#libraryGrid .lib-pick').prop('checked', false).trigger('change');
});

$('#libApply').on('click', function(){
  const picks = [];
  $('#libraryGrid .library-item').each(function(){
    const $it = $(this);
    const $cb = $it.find('.lib-pick');
    if ($cb.is(':checked')) {
      const url = $cb.data('url');
      const title = $it.find('.title-input').val().trim();
      picks.push({url, title});
    }
  });

  if (LIB_MODE === 'main') {
    if (picks.length!==1) return modalUtil.alert('נא לבחור תמונה אחת','error');
    const p = picks[0];
    const $wrap = $('#mainChosenFromLib').empty();
    const $t = $(`
      <div class="thumb m-1">
        <span class="remove" title="מחק">×</span>
        <img src="${p.url}" style="height:80px" alt="">
        <div class="title-under">
          <input type="text" class="form-control form-control-sm"
                 name="main_img_title_lib" value="${escapeHtml(p.title)}" placeholder="כותרת לתמונה">
        </div>
      </div>
    `);
    $wrap.append($t);
    $('#mainImgUrlLib').val(p.url);
    $('#mainImgCropped').val('');
    $('#mainImgUrl').val('');
    $('#mainImgTitleInput').val('');
  } else {
    const currentCount = totalGalleryCount();
    const remaining = Math.max(GALLERY_MAX - currentCount, 0);
    if (picks.length > remaining) {
      modalUtil.alert(`ניתן להוסיף עוד ${remaining} תמונות בלבד (מקסימום ${GALLERY_MAX}).`, 'error');
    }
    const toAdd = picks.slice(0, remaining);
    const wrap = $('#newGallery');
    toAdd.forEach(p=>{
      const $t = $(`
        <label class="thumb m-1">
          <span class="remove" title="מחק">×</span>
          <span class="drag-handle" title="גרור">≡</span>
          <img src="${p.url}" style="height:80px" alt="">
          <input type="hidden" class="lib-added" value="${p.url}">
          <div class="title-under">
            <input type="text" class="form-control form-control-sm gallery-title-input"
                   value="${escapeHtml(p.title)}" placeholder="כותרת לתמונה">
          </div>
        </label>`);
      wrap.append($t);
    });
    updateOrder();
    updateGalleryMeta();
  }
  $('#libraryModal').fadeOut(120);
});

$(document).on('click','#mainChosenFromLib .remove', function(){
  $(this).closest('.thumb').remove();
  $('#mainImgUrlLib').val('');
});

function escapeHtml(s){return s.replace(/[&<>"']/g,m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m]));}

/* drag&drop uploader */
const libUploader = document.getElementById('libUploader');
const libFilesInp = document.getElementById('libFiles');
const libUploadBtn = document.getElementById('libUploadBtn');
const libStatus    = document.getElementById('libUploadStatus');
const libraryGrid  = document.getElementById('libraryGrid');

['dragenter','dragover'].forEach(ev=>{
  libUploader && libUploader.addEventListener(ev, e=>{ e.preventDefault(); e.stopPropagation(); libUploader.classList.add('drag'); });
});
['dragleave','drop'].forEach(ev=>{
  libUploader && libUploader.addEventListener(ev, e=>{ e.preventDefault(); e.stopPropagation(); libUploader.classList.remove('drag'); });
});
libUploader && libUploader.addEventListener('drop', e=>{
  const dt = e.dataTransfer;
  if (!dt || !dt.files || !dt.files.length) return;
  libFilesInp.files = dt.files;
});

libUploadBtn && libUploadBtn.addEventListener('click', async ()=>{
  if (!libFilesInp.files || !libFilesInp.files.length) {
    libStatus.textContent = 'לא נבחרו קבצים.';
    return;
  }
  try {
    libStatus.textContent = 'מעלה...';
    const fd = new FormData();
    for (const f of libFilesInp.files) fd.append('files[]', f);

    const resp = await fetch('add-product.php?action=upload_to_library', { method: 'POST', body: fd });
    const data = await resp.json();
    if (!data.ok) throw new Error(data.msg || 'שגיאה בהעלאה');

    // If modal open, prepend new items using their thumbs
    if ($('#libraryModal').is(':visible')) {
      const frag = document.createDocumentFragment();
      (data.items||[]).forEach(it=>{
        if (!it.ok) return;
        const thumb = it.thumb || it.url;
        const el = document.createElement('label');
        el.className = 'library-item';
        el.innerHTML = `
            <input type="checkbox" class="lib-pick" data-url="${it.url}">
            <img src="${thumb}" alt="">
            <div class="item-name" title="${it.name}">${it.name}</div>
            <input type="text" class="form-control form-control-sm title-input" placeholder="כותרת" value="${it.title}">
        `;
        frag.appendChild(el);
      });
      libraryGrid.prepend(frag);
    }
    libFilesInp.value = '';
    libStatus.textContent = 'העלאה הושלמה.';
  } catch (e) {
    console.error(e);
    libStatus.textContent = 'אירעה שגיאה בהעלאה.';
  } finally {
    setTimeout(()=>{ libStatus.textContent = ''; }, 2500);
  }
});

/* ==========================================================
   GALLERY handlers
   ========================================================== */
function totalGalleryCount(){
  return $('#oldGallery img').length + $('#newGallery img').length;
}

$(document).on('click','.thumb .remove',function(){
  const $thumb = $(this).closest('.thumb');
  const img=$thumb.find('input[type="hidden"]').val() || $thumb.find('img').attr('src');
  if ($thumb.closest('#oldGallery').length) {
    $('<input>',{type:'hidden',name:'delete_images[]',value:img}).appendTo('#productForm');
  }
  $thumb.remove();
  updateOrder();
  updateGalleryMeta();
});

$('#galleryInput').on('change',function(){
  const wrap=$('#newGallery');
  const files=[...this.files];

  const currentCount = totalGalleryCount();
  const remaining = Math.max(GALLERY_MAX - currentCount, 0);
  if (files.length > remaining) {
    modalUtil.alert(`ניתן להוסיף עוד ${remaining} תמונות בלבד (מקסימום ${GALLERY_MAX}).`, 'error');
  }
  const slice = files.slice(0, remaining);

  slice.forEach((f)=>{
    const r=new FileReader();
    r.onload=e=>{
      const $t=$(`
        <label class="thumb m-1">
          <span class="remove" title="מחק">×</span>
          <span class="drag-handle" title="גרור">≡</span>
          <img src="${e.target.result}" style="height:80px" alt="">
          <div class="title-under">
            <input type="text" class="form-control form-control-sm gallery-title-input"
                   placeholder="כותרת לתמונה">
          </div>
        </label>`);
      wrap.append($t);
      updateOrder();
      updateGalleryMeta();
    };
    r.readAsDataURL(f);
  });

  this.value = '';
});

/* Sortable with handle */
;['oldGallery','newGallery'].forEach(id=>{
  const el=document.getElementById(id);
  if(el) new Sortable(el,{
    animation:150,
    handle: '.drag-handle',
    onSort: function(){ updateOrder(); updateGalleryMeta(); }
  });
});

function updateOrder(){
  const imgs=$('#oldGallery img,#newGallery img').map((_,im)=>$(im).attr('src')).get();
  $('#gallerySort').val(JSON.stringify(imgs));
}

function collectGalleryTitles() {
  const titleMap = {};
  $('#oldGallery .thumb').each(function(){
    const url = $(this).find('img').attr('src');
    const t = $(this).find('.existing-gallery-title').val() || '';
    titleMap[url] = t;
  });
  $('#newGallery .thumb').each(function(){
    const url = $(this).find('img').attr('src');
    const t = $(this).find('.gallery-title-input').val() || '';
    titleMap[url] = t;
  });

  let order = [];
  try { order = JSON.parse($('#gallerySort').val()||'[]'); } catch(e){ order=[]; }
  const out = order.map(u => ({url:u, title: titleMap[u] || ''}));
  $('#galleryTitlesJson').val(JSON.stringify(out));
}

function updateGalleryMeta(){
  const count = totalGalleryCount();
  $('#galleryCountBadge').text(`סה"כ ${count}/${GALLERY_MAX}`).show();
  $('#summaryGallery').text(`תמונות בגלריה: ${count}`);
}

/* ==========================================================
   PRICE rows + basic validation hints
   ========================================================== */
function priceRowTemplate(){
  return `
    <tr>
      <td><input type="number" step="any" name="weights[]" class="form-control" placeholder="0"></td>
      <td>
        <select name="units[]" class="form-control">
          <option value="kg">Kg</option>
          <option value="ton">Ton</option>
          <option value="g">Grams</option>
          <option value="unit">Unit</option>
        </select>
      </td>
      <td><input type="number" step="0.01" name="prices[]" class="form-control" placeholder="0.00"></td>
      <td><input type="text" name="barcodes[]" class="form-control" placeholder="1234-5678"></td>
      <td><button type="button" class="btn btn-danger btn-sm remove-row" aria-label="מחק">×</button></td>
    </tr>`;
}

$('#addPriceRow').click(() => {
  $('#priceTable tbody').append(priceRowTemplate());
  updateSummary();
});
$(document).on('click', '.remove-row', function () {
  $(this).closest('tr').remove();
  updateSummary();
});

function scanPriceIssues(){
  const rows = $('#priceTable tbody tr');
  let emptyPrice = 0;
  let emptyWeight = 0;
  let dupBarcodes = 0;

  const barcodes = {};
  rows.each(function(){
    const price = $(this).find('input[name="prices[]"]').val();
    const weight = $(this).find('input[name="weights[]"]').val();
    const bc = ($(this).find('input[name="barcodes[]"]').val()||'').trim();
    if (!price) emptyPrice++;
    if (!weight) emptyWeight++;
    if (bc) {
      if (barcodes[bc]) dupBarcodes++;
      barcodes[bc] = true;
    }
  });

  const msgs = [];
  if (emptyPrice) msgs.push(`${emptyPrice} מחירים ריקים`);
  if (emptyWeight) msgs.push(`${emptyWeight} כמויות ריקות`);
  if (dupBarcodes) msgs.push(`יש כפילויות ברקוד`);
  const $el = $('#priceIssues');
  if (msgs.length) { $el.text('בדיקה: ' + msgs.join(' · ')).show(); } else { $el.hide(); }
}

/* ==========================================================
   Simple modal bg click
   ========================================================== */
(function ensureModalCloseOnBg(){
  $('#libraryModal').on('click', function(e){
    if (e.target === this) $(this).fadeOut(120);
  });
  $('#cropModal').on('click', function(e){
    if (e.target === this) $(this).fadeOut(120);
  });
})();

/* ==========================================================
   SUMMARY BAR + SUBMIT
   ========================================================== */
function updateSummary(){
  const catText = $('#main_id option:selected').text() || '—';
  const subText = $('#model_id option:selected').text() || '—';
  const priceRows = $('#priceTable tbody tr').length;
  $('#summaryCat').text('קטגוריה: ' + (catText==='– בחר –'?'—':catText));
  $('#summarySub').text('משנית: ' + (subText==='– בחר –'?'—':subText));
  $('#summaryPrices').text('שורות מחירים: ' + priceRows);
  scanPriceIssues();
}

$('#productForm').on('submit', function(e){
  updateOrder();
  collectGalleryTitles();

  if (totalGalleryCount() > GALLERY_MAX) {
    e.preventDefault();
    return modalUtil.alert(`מקסימום ${GALLERY_MAX} תמונות בגלריה.`, 'error');
  }

  if ($('input[name="main_img_source"]:checked').val()==='library') {
    const libTitle = $('input[name="main_img_title_lib"]').val() || '';
    if (!$('input[name="main_img_title"]').length) {
      $('<input>',{type:'hidden',name:'main_img_title',value:libTitle}).appendTo(this);
    } else {
      $('input[name="main_img_title"]').val(libTitle);
    }
  }
});

/* ==========================================================
   INIT
   ========================================================== */
function init(){
  fetchCatLists(()=>{
    const catVal = $('#main_id').val();
    if (catVal) { renderSubs(catVal, $('#model_id').val()); $('#model_id,#subQuickFilter').prop('disabled', false); }
    validateStep1();
    updateSummary();
    updateGalleryMeta();
  });
  restoreTabs();
}
$(init);

$('#wizardTabs a[data-toggle="tab"]').on('click', function(){
  localStorage.setItem(WIZARD_KEY, $(this).attr('href'));
});
</script>
<script src="app/js/main.js"></script>
<script src="app/js/jquery-ui.js"></script>
<script src="app/js/dashboard.js"></script>
</body>
</html>
