<?php
declare(strict_types=1);
error_reporting(E_ALL);
ini_set('display_errors', '1');
header('Content-Type: application/json; charset=utf-8');

function jexit(array $payload): void {
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

/**
 * Make sure this include sets $conn as a MySQLi connection.
 * Adjust the path to whatever bootstrap/config file your project uses.
 */
$bootPaths = [
  __DIR__ . '/../includes/Conn.php',
];
foreach ($bootPaths as $p) { if (is_file($p)) { require_once $p; break; } }
if (!isset($conn) || !($conn instanceof mysqli)) {
  jexit(['ok' => false, 'error' => 'DB connection ($conn) not available in endpoint.']);
}

/** Load your product box renderer if not already loaded */
if (!function_exists('renderProductBoxCarStyle')) {
  $pbCandidates = [
    __DIR__ . '/../includes/product_box.php',
    __DIR__ . '/../product_box.php',
  ];
  foreach ($pbCandidates as $pp) { if (is_file($pp)) { require_once $pp; break; } }
}

/** Params */
$catId  = isset($_GET['category_id']) ? (int)$_GET['category_id'] : 0;
$offset = isset($_GET['offset']) ? max(0, (int)$_GET['offset']) : 0;
$limit  = isset($_GET['limit']) ? max(1, min(100, (int)$_GET['limit'])) : 10;

if ($catId <= 0) jexit(['ok' => false, 'error' => 'Invalid or missing category_id.']);

/** Query (fetch limit+1 to detect "has more") */
$fetchLimit = $limit + 1;
$sql = "
  SELECT p.id, p.title_he, p.title_ar, p.title, p.images, p.category,
         sc.category_name AS category_name
  FROM shop_products AS p
  LEFT JOIN shop_categories AS sc ON sc.id = p.category
  WHERE p.status = 'publish' AND p.category = ?
  ORDER BY p.id DESC
  LIMIT ? OFFSET ?
";
$stmt = $conn->prepare($sql);
if (!$stmt) jexit(['ok' => false, 'error' => 'Query prepare failed.']);
$stmt->bind_param('iii', $catId, $fetchLimit, $offset);
$stmt->execute();
$res = $stmt->get_result();

$rows = [];
while ($row = $res->fetch_assoc()) $rows[] = $row;
$stmt->close();

$hasMore = false;
if (count($rows) > $limit) {
  $hasMore = true;
  array_pop($rows); // keep exactly $limit to render
}

/** Render HTML using your existing renderer */
ob_start();
if ($rows) {
  if (function_exists('renderProductBoxCarStyle')) {
    foreach ($rows as $prod) {
      $title = $prod['title_he'] ?: ($prod['title'] ?? '');
      renderProductBoxCarStyle($conn, $prod, 'default', $prod['category_name'] ?? '');
    }
  } else {
    // Fallback (only if your renderer is missing)
    foreach ($rows as $prod) {
      $title = htmlspecialchars($prod['title_he'] ?: ($prod['title'] ?? ''), ENT_QUOTES, 'UTF-8');
      echo '<div class="card"><div class="card-body">'.$title.'</div></div>';
    }
  }
}
$html = ob_get_clean();

jexit([
  'ok'         => true,
  'html'       => $html,
  'appended'   => count($rows),
  'next_offset'=> $offset + count($rows),
  'has_more'   => $hasMore,
]);
