<?php
/**
 * includes/image_utils.php
 * Utilities for high-quality image processing using Imagick (if available) with GD fallback.
 */
declare(strict_types=1);

function imgutils_supported(): array {
    return ['imagick' => class_exists('Imagick'), 'gd' => function_exists('imagecreatetruecolor')];
}
function _ensure_dir(string $absDir): bool { return is_dir($absDir) ?: @mkdir($absDir, 0775, true); }
function _ext_from_mime(string $mime): string {
    switch (strtolower($mime)) {
        case 'image/jpeg': case 'image/jpg': return 'jpg';
        case 'image/png':  return 'png';
        case 'image/gif':  return 'gif';
        case 'image/webp': return 'webp';
        default: return 'jpg';
    }
}
function imgutils_save_upload(array $file, string $absDir, string $relDir): array {
    if (empty($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) return ['ok'=>false,'msg'=>'No upload'];
    $info = getimagesize($file['tmp_name']); if (!$info) return ['ok'=>false,'msg'=>'Invalid image'];
    $ext = _ext_from_mime($info['mime'] ?? ''); if (!_ensure_dir($absDir)) return ['ok'=>false,'msg'=>'Cannot create dir'];
    $name = uniqid('img_', true) . ".$ext"; $abs = rtrim($absDir,'/')."/".$name; $rel=rtrim($relDir,'/')."/".$name;
    if (!move_uploaded_file($file['tmp_name'], $abs)) return ['ok'=>false,'msg'=>'Failed to move upload'];
    @chmod($abs, 0664); return ['ok'=>true,'abs'=>$abs,'rel'=>$rel,'width'=>$info[0]??0,'height'=>$info[1]??0,'mime'=>$info['mime']??''];
}
function imgutils_pad_to_square(string $srcAbs, string $dstAbs, string $bgHex='#FFFFFF'): array {
    $s=imgutils_supported();
    if($s['imagick']){ try{ $im=new Imagick($srcAbs); $w=$im->getImageWidth(); $h=$im->getImageHeight(); $size=max($w,$h);
        $c=new Imagick(); $c->newImage($size,$size,new ImagickPixel($bgHex),$im->getImageFormat());
        $x=intval(($size-$w)/2); $y=intval(($size-$h)/2); $c->compositeImage($im, Imagick::COMPOSITE_DEFAULT, $x,$y);
        $c->setImageCompressionQuality(92); $c->writeImage($dstAbs); $im->destroy(); $c->destroy(); return ['ok'=>true];
    }catch(Throwable $e){return ['ok'=>false,'msg'=>$e->getMessage()];}}
    elseif($s['gd']){ $info=getimagesize($srcAbs); if(!$info) return ['ok'=>false,'msg'=>'Bad image'];
        [$w,$h]=$info; $size=max($w,$h); $dst=imagecreatetruecolor($size,$size); $white=imagecolorallocate($dst,255,255,255);
        imagefilledrectangle($dst,0,0,$size,$size,$white);
        switch($info['mime']){ case 'image/png': $src=imagecreatefrompng($srcAbs); break; case 'image/gif': $src=imagecreatefromgif($srcAbs); break; case 'image/webp': $src=function_exists('imagecreatefromwebp')?imagecreatefromwebp($srcAbs):null; break; default: $src=imagecreatefromjpeg($srcAbs); }
        if(!$src) return ['ok'=>false,'msg'=>'Unsupported']; $x=intval(($size-$w)/2); $y=intval(($size-$h)/2);
        imagecopy($dst,$src,$x,$y,0,0,$w,$h); $ok=imagejpeg($dst,$dstAbs,92); imagedestroy($src); imagedestroy($dst); return ['ok'=>$ok];
    }
    return ['ok'=>false,'msg'=>'No library'];
}
function imgutils_crop(string $srcAbs, string $dstAbs, int $x,int $y,int $w,int $h): array {
    $s=imgutils_supported();
    if($s['imagick']){ try{ $im=new Imagick($srcAbs); $im->cropImage($w,$h,$x,$y); $im->setImageCompressionQuality(92); $im->writeImage($dstAbs); $im->destroy(); return ['ok'=>true]; } catch(Throwable $e){return ['ok'=>false,'msg'=>$e->getMessage()];} }
    elseif($s['gd']){ $info=getimagesize($srcAbs); if(!$info) return ['ok'=>false,'msg'=>'Bad image'];
        switch($info['mime']){ case 'image/png': $src=imagecreatefrompng($srcAbs); break; case 'image/gif': $src=imagecreatefromgif($srcAbs); break; case 'image/webp': $src=function_exists('imagecreatefromwebp')?imagecreatefromwebp($srcAbs):null; break; default: $src=imagecreatefromjpeg($srcAbs); }
        if(!$src) return ['ok'=>false,'msg'=>'Unsupported']; $dst=imagecreatetruecolor($w,$h); imagecopy($dst,$src,0,0,$x,$y,$w,$h); $ok=imagejpeg($dst,$dstAbs,92); imagedestroy($src); imagedestroy($dst); return ['ok'=>$ok];
    }
    return ['ok'=>false,'msg'=>'No library'];
}
function imgutils_resize(string $srcAbs,string $dstAbs,int $tw,int $th,bool $cover=true,?string $padHex=null): array {
    $s=imgutils_supported();
    if($s['imagick']){ try{ $im=new Imagick($srcAbs); $im->setImageCompressionQuality(92); $im->setImageInterlaceScheme(Imagick::INTERLACE_JPEG);
        $w=$im->getImageWidth(); $h=$im->getImageHeight(); $sr=$w/max(1,$h); $dr=$tw/max(1,$th);
        if($cover){ if($sr>$dr){$nh=$th;$nw=intval($th*$sr);} else {$nw=$tw;$nh=intval($tw/$sr);} $im->resizeImage($nw,$nh,Imagick::FILTER_LANCZOS,1); $im->cropImage($tw,$th,intval(($nw-$tw)/2),intval(($nh-$th)/2)); }
        else { $im->resizeImage($tw,$th,Imagick::FILTER_LANCZOS,1,true); if($padHex){ $c=new Imagick(); $c->newImage($tw,$th,new ImagickPixel($padHex),$im->getImageFormat()); $x=intval(($tw-$im->getImageWidth())/2); $y=intval(($th-$im->getImageHeight())/2); $c->compositeImage($im,Imagick::COMPOSITE_DEFAULT,$x,$y); $im=$c; } }
        $im->writeImage($dstAbs); $im->destroy(); return ['ok'=>true];
    }catch(Throwable $e){return ['ok'=>false,'msg'=>$e->getMessage()];} }
    elseif($s['gd']){ $info=getimagesize($srcAbs); if(!$info) return ['ok'=>false,'msg'=>'Bad image']; [$sw,$sh]=$info;
        switch($info['mime']){ case 'image/png': $src=imagecreatefrompng($srcAbs); break; case 'image/gif': $src=imagecreatefromgif($srcAbs); break; case 'image/webp': $src=function_exists('imagecreatefromwebp')?imagecreatefromwebp($srcAbs):null; break; default: $src=imagecreatefromjpeg($srcAbs); }
        if(!$src) return ['ok'=>false,'msg'=>'Unsupported']; $dst=imagecreatetruecolor($tw,$th);
        if(!$cover && $padHex){ $col=imagecolorallocate($dst,255,255,255); imagefilledrectangle($dst,0,0,$tw,$th,$col); }
        $sr=$sw/max(1,$sh); $dr=$tw/max(1,$th);
        if($cover){ if($sr>$dr){$nh=$th;$nw=intval($th*$sr);} else {$nw=$tw;$nh=intval($tw/$sr);} $tmp=imagecreatetruecolor($nw,$nh); imagecopyresampled($tmp,$src,0,0,0,0,$nw,$nh,$sw,$sh); imagecopy($dst,$tmp,intval(($tw-$nw)/2),intval(($th-$nh)/2),intval(($nw-$tw)/2),intval(($nh-$th)/2),$tw,$th); imagedestroy($tmp); }
        else { if($sr>$dr){$nw=$tw;$nh=intval($tw/$sr);} else {$nh=$th;$nw=intval($th*$sr);} $tmp=imagecreatetruecolor($nw,$nh); imagecopyresampled($tmp,$src,0,0,0,0,$nw,$nh,$sw,$sh); imagecopy($dst,$tmp,intval(($tw-$nw)/2),intval(($th-$nh)/2),0,0,$nw,$nh); imagedestroy($tmp); }
        $ok=imagejpeg($dst,$dstAbs,92); imagedestroy($src); imagedestroy($dst); return ['ok'=>$ok];
    }
    return ['ok'=>false,'msg'=>'No library'];
}
function imgutils_generate_variants(string $absDir,string $relDir,string $srcRelBasename,bool $padIfNotSquare=true): array {
    $srcAbs=rtrim($absDir,'/')."/".$srcRelBasename; $nameNoExt=preg_replace('/\.[^.]+$/','',$srcRelBasename); $ext=strtolower(pathinfo($srcRelBasename, PATHINFO_EXTENSION));
    $thumb226=$nameNoExt.'_226x226.'.$ext; $tiny10=$nameNoExt.'_10x10.'.$ext;
    $abs226=rtrim($absDir,'/')."/".$thumb226; $abs10=rtrim($absDir,'/')."/".$tiny10;
    if($padIfNotSquare){ $tmp=rtrim($absDir,'/')."/".$nameNoExt.'_square.'.$ext; $pad=imgutils_pad_to_square($srcAbs,$tmp,'#FFFFFF'); if(($pad['ok']??false)) $srcAbs=$tmp; }
    $r1=imgutils_resize($srcAbs,$abs226,226,226,true); $r2=imgutils_resize($srcAbs,$abs10,10,10,true);
    return ['ok'=>($r1['ok']??false)&&($r2['ok']??false), 'thumb_226'=>rtrim($relDir,'/')."/".$thumb226, 'tiny_10'=>rtrim($relDir,'/')."/".$tiny10];
}
