<?php
$internal = true;

error_reporting(E_ALL);
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');

require_once __DIR__ . "/includes/Conn.php"; // must define $conn (mysqli)
if (!($conn instanceof mysqli)) { die("DB connection not available."); }

/* ========================= Helpers ========================= */
if (!function_exists('h')) {
  function h($s): string {
    return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8', false);
  }
}
function starts_with($haystack, $needle){ return strncmp($haystack, $needle, strlen($needle)) === 0; }
function is_url_or_root($p){ return preg_match('#^https?://#i', $p) || strncmp($p, '/', 1) === 0; }
function slugify_he($txt){
  $t = preg_replace('~[^\pL\d]+~u', '-', (string)$txt);
  $t = trim($t, '-');
  $t = preg_replace('~-+~', '-', $t);
  return $t ?: 'cat';
}

/* ===== image helpers for product box (if used internally) ===== */
if (!function_exists('product_thumb_url')) {
  function product_thumb_url(?string $raw): string {
    $placeholder = '/assets/images/dashboard/default-product.jpg';
    if (!$raw) return $placeholder;
    $url = '/' . ltrim($raw, '/');
    $rel = ltrim($url, '/');
    if (strpos($rel, 'uploads/library/') === 0) {
      $dir = dirname($rel);
      $file = basename($rel);
      $thumbRel = $dir . '/_thumbs/' . $file;
      $thumbAbs = $_SERVER['DOCUMENT_ROOT'] . '/' . $thumbRel;
      if (is_file($thumbAbs)) return '/' . $thumbRel;
    }
    return $url;
  }
}

/* ====== bring in your exact product box (same as slider) ====== */
if (!function_exists('renderProductBoxCarStyle')) {
  $productBox = __DIR__ . '/includes/product_box.php';
  if (is_file($productBox)) require_once $productBox;
}

/* ===================== Config ===================== */
$ICONS_BASE  = '/assets/svg/';
$ICON_MAP = [
  'בעלי כנף' => 'bird.svg',
  'חתולים'   => 'cat.svg',
  'כלבים'    => 'dog.svg',
  'סוסים'    => 'horse.svg',
  'פרות'     => 'cow.svg',
  'עגלים'    => 'cow.svg',
  'תרנגולות' => 'chicken.svg',
  'עזים'     => 'goat.svg',
];
function resolve_icon_url($catName, $dbIcon, $ICONS_BASE, $ICON_MAP){
  if (!empty($dbIcon)) return is_url_or_root($dbIcon) ? $dbIcon : $ICONS_BASE . ltrim($dbIcon, '/');
  if (!empty($ICON_MAP[$catName])) {
    $m = $ICON_MAP[$catName]; return is_url_or_root($m) ? $m : $ICONS_BASE . ltrim($m, '/');
  }
  $slug = strtolower(slugify_he($catName));
  if (!empty($ICON_MAP[$slug])) {
    $m = $ICON_MAP[$slug]; return is_url_or_root($m) ? $m : $ICONS_BASE . ltrim($m, '/');
  }
  $candidate = $ICONS_BASE . strtolower(slugify_he($catName)) . '.svg';
  return $candidate ?: ($ICONS_BASE . 'folder.svg');
}

/* ===================== Inputs ===================== */
$settings     = $settings ?? [];
$limitDefault = (int)($settings['cards_per_tab'] ?? 12);

$view   = $_GET['view'] ?? 'grid';
$view   = in_array($view, ['grid','list'], true) ? $view : 'grid';
$sort   = $_GET['sort'] ?? 'newest';
$sort   = in_array($sort, ['newest','price-asc','price-desc'], true) ? $sort : 'newest';
$page   = max(1, (int)($_GET['page'] ?? 1));
$limit  = max(1, min(100, (int)($_GET['limit'] ?? $limitDefault))); // allow 100 like template
$offset = ($page - 1) * $limit;

$category_id   = (int)($_GET['category_id'] ?? 0);
$category_name = ''; // for meta/title updates

/* =================== Categories (ALL) =================== */
$cats = [];
$resCats = $conn->query("
  SELECT id, category_name AS name, COALESCE(NULLIF(icon,''),'') AS icon
  FROM shop_categories
  WHERE COALESCE(is_active,1)=1
  ORDER BY sort_order ASC, category_name ASC
");
while ($row = $resCats->fetch_assoc()) $cats[] = $row;
$resCats->free();

foreach ($cats as &$c) {
  $c['icon_url'] = resolve_icon_url($c['name'], $c['icon'], $ICONS_BASE, $ICON_MAP);
  if ($category_id && (int)$c['id'] === $category_id) $category_name = $c['name'];
}
unset($c);

if (!$category_name && $category_id) {
  $rsn = $conn->prepare("SELECT category_name FROM shop_categories WHERE id=? LIMIT 1");
  $rsn->bind_param('i', $category_id);
  $rsn->execute();
  $tmp = $rsn->get_result()->fetch_assoc();
  $category_name = $tmp['category_name'] ?? '';
  $rsn->close();
}

/* ===================== Build SQL (products) ===================== */
$where = ["p.status = 'publish'"];
$types = '';
$params = [];

if ($category_id > 0) {
  $where[] = "p.category = ?";
  $types  .= 'i';
  $params[] = $category_id;
}

$joinMinPrice = "LEFT JOIN (
  SELECT product_id, MIN(price) AS min_price
  FROM shop_product_prices
  GROUP BY product_id
) mp ON mp.product_id = p.id";

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

switch ($sort) {
  case 'price-asc':
    $orderBy = "CASE WHEN mp.min_price IS NULL THEN 1 ELSE 0 END, mp.min_price ASC, p.id DESC";
    break;
  case 'price-desc':
    $orderBy = "CASE WHEN mp.min_price IS NULL THEN 1 ELSE 0 END, mp.min_price DESC, p.id DESC";
    break;
  case 'newest':
  default:
    $orderBy = "p.id DESC";
    break;
}

/* count */
$countSql = "
  SELECT COUNT(*) AS cnt
  FROM (
    SELECT p.id
    FROM shop_products p
    $joinMinPrice
    $whereSql
    GROUP BY p.id
  ) x
";
$countStmt = $conn->prepare($countSql);
if ($types) $countStmt->bind_param($types, ...$params);
$countStmt->execute();
$totalRows = (int)$countStmt->get_result()->fetch_assoc()['cnt'];
$totalPages = max(1, (int)ceil($totalRows / $limit));

$fromRow = $totalRows ? ($offset + 1) : 0;
$toRow   = min($offset + $limit, $totalRows);

/* page rows */
$dataSql = "
  SELECT
    p.id, p.title_he, p.title_ar, p.ingredients_he, p.ingredients_ar, p.images, p.category,
    c.category_name AS category_name,
    mp.min_price
  FROM shop_products p
  LEFT JOIN shop_categories c ON c.id = p.category
  $joinMinPrice
  $whereSql
  GROUP BY p.id
  ORDER BY $orderBy
  LIMIT ? OFFSET ?
";
$dataTypes = $types . 'ii';
$dataParams = $params;
$dataParams[] = $limit;
$dataParams[] = $offset;

$dataStmt = $conn->prepare($dataSql);
$dataStmt->bind_param($dataTypes, ...$dataParams);
$dataStmt->execute();
$resProducts = $dataStmt->get_result();
$products = [];
while ($row = $resProducts->fetch_assoc()) $products[] = $row;

/* ========= fragment render for AJAX ========= */
function render_products_fragment(mysqli $conn, array $products, string $view, int $page, int $totalPages, int $totalRows, int $fromRow, int $toRow): void { ?>
  <div class="listing-metadata" data-total="<?= (int)$totalRows ?>" data-from="<?= (int)$fromRow ?>" data-to="<?= (int)$toRow ?>"></div>

  <div class="row">
    <div class="col-lg-12 listing-list-car-wrap listing-grid-car-wrap">
      <div class="flat-tabs themesflat-tabs category-filter">
        <div class="content-tab">
          <div class="content-inner tab-content">
            <div class="list-car-list-1<?= $view==='grid' ? ' list-car-grid-1' : '' ?>">
              <?php if (!empty($products)): ?>
                <?php foreach ($products as $prod): ?>
                  <?php renderProductBoxCarStyle($conn, $prod, 'style-2', $prod['category_name'] ?? ''); ?>
                <?php endforeach; ?>
              <?php else: ?>
                <p>לא נמצאו מוצרים תואמים.</p>
              <?php endif; ?>
            </div>
          </div>
        </div>

        <?php if ($totalPages > 1): ?>
          <div class="themesflat-pagination pagination-style1 clearfix center mt-30">
            <ul>
              <?php if ($page > 1): ?>
                <li><a href="#" class="page-numbers style ajax-page-link" data-page="<?= $page - 1 ?>"><i class="far fa-angle-left"></i></a></li>
              <?php endif; ?>

              <?php
              $start = max(1, $page - 2);
              $end   = min($totalPages, $page + 2);
              for ($i = $start; $i <= $end; $i++): ?>
                <li>
                  <a href="#" class="page-numbers ajax-page-link<?= $i===$page ? ' current' : '' ?>" data-page="<?= $i ?>"><?= $i ?></a>
                </li>
              <?php endfor; ?>

              <?php if ($page < $totalPages): ?>
                <li><a href="#" class="page-numbers style ajax-page-link" data-page="<?= $page + 1 ?>"><i class="far fa-angle-left"></i></a></li>
              <?php endif; ?>
            </ul>
          </div>
        <?php endif; ?>

      </div>
    </div>
  </div>
<?php }

/* ===== if AJAX request: output fragment only and exit ===== */
$isAjax = (isset($_GET['ajax']) && $_GET['ajax'] == '1')
  || (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH'])==='xmlhttprequest');

if ($isAjax) {
  render_products_fragment($conn, $products, $view, $page, $totalPages, $totalRows, $fromRow, $toRow);
  exit;
}

/* ======== SEO defaults (updated dynamically on category click) ======== */
$baseTitle   = 'רשימת מוצרים';
$pageTitle   = $category_name ? ('מוצרים: ' . $category_name) : $baseTitle;
$description = $category_name ? ('מוצרים בקטגוריית ' . $category_name) : 'מוצרים למכירה לפי קטגוריה';
$keywords    = $category_name ? ($category_name.', מוצרים, קטגוריות, מחירים') : 'מוצרים, קטגוריות, מחירים';
?>
<!DOCTYPE html>
<html lang="he" dir="rtl">
<head>
  <meta charset="utf-8">
  <title id="dynamic-title"><?= h($pageTitle) ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">

  <!-- CSS -->
  <link rel="stylesheet" href="/app/css/font-awesome.css">
  <link rel="stylesheet" href="/app/css/styles.css">
  <link rel="shortcut icon" href="/assets/images/logo/Favicon.png">
  <link rel="/apple-touch-icon-precomposed" href="/assets/images/logo/Favicon.png">

  <!-- SEO (will also be updated by JS on category click) -->
  <meta name="description" content="<?= h($description) ?>" id="meta-desc">
  <meta name="keywords"    content="<?= h($keywords) ?>" id="meta-keys">
  <meta property="og:title"       content="<?= h($pageTitle) ?>" id="meta-og-title">
  <meta property="og:description" content="<?= h($description) ?>" id="meta-og-desc">
  <meta property="og:url"         content="<?= 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] ?>" id="meta-og-url">

  <style>
    /* ===== Icon cubes row ===== */
   
  </style>
</head>
<body class="body header-fixed">
<div id="wrapper">
  <div id="pagee" class="clearfix">
    <?php include './includes/header.php'; ?>

 

    <!-- ================= LISTING GRID – PRODUCTS ================= -->
    <section class="listing-grid tf-section3">
      <div class="container">

     

        <!-- Icon cubes (ALL categories, plus "All") -->
        <div class="row" id="cat-tabs">
          <div class="col-12">
            <nav class="cat-container" aria-label="קטגוריות מוצרים">
              <div class="cat-scroll">
                <div class="cat-track">
                  <!-- All categories cube -->
                  <div class="cat-wrap">
                    <button
                      type="button"
                      class="cat-card<?= $category_id===0 ? ' active' : '' ?>"
                      data-category-id="0"
                      data-category-name="">
                      <span class="cat-inner">
                        <span class="cat-label">כל הקטגוריות</span>
                        <img src="<?= h($ICONS_BASE) ?>custom-blend.svg" alt="כל הקטגוריות" class="cat-icon" loading="lazy">
                      </span>
                    </button>
                    <a class="btn btn-sm btn-primary view-all" tabindex="-1">צפה הכל</a>
                  </div>

                  <?php foreach ($cats as $i => $c):
                    $isActive = ($category_id === (int)$c['id']);
                  ?>
                  <div class="cat-wrap">
                    <button
                      type="button"
                      class="cat-card<?= $isActive ? ' active' : '' ?>"
                      data-category-id="<?= (int)$c['id'] ?>"
                      data-category-name="<?= h($c['name']) ?>">
                      <span class="cat-inner">
                        <span class="cat-label"><?= h($c['name']) ?></span>
                        <img src="<?= h($c['icon_url']) ?>" alt="<?= h($c['name']) ?>" class="cat-icon" loading="lazy"
                             onerror="this.onerror=null;this.src='<?= h($ICONS_BASE) ?>folder.svg';">
                      </span>
                    </button>
                    <a class="btn btn-sm btn-primary view-all" tabindex="-1">צפה הכל</a>
                  </div>
                  <?php endforeach; ?>
                </div>
              </div>
            </nav>
          </div>
        </div>
   <!-- ===== Heading: your exact structure ===== -->
        <div class="row">
          <div class="col-lg-12">
            <div class="category-filter flex justify-space align-center mb-30 flex-wrap gap-8">
              <div class="box-1 flex align-center flex-wrap gap-8">
               <!-- Section Title -->
    <div class="heading-section">
      <h2 id="page-title"><?= h($category_name ?: 'קטלוג מוצרים בר שרבלה') ?></h2>
        <div class="title-inner style">
              <div class="title-group fs-12">
                <a class="home fw-6 text-color-3" href="/">דף הבית</a>
                <span id="breadcrumb-leaf"> / <?= h($category_name) ?> (<span id="results-counter"><?= (int)$totalRows ?></span> מוצרים)</span>
              </div>
            </div>
  
    </div>

              </div>
              <div class="box-2 flex flex-wrap gap-8">
                <a href="#" class="btn-view grid<?= $view==='grid' ? ' active' : '' ?>" id="view-grid">
                  <svg width="25" height="25" viewBox="0 0 25 25" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M5.04883 6.40508C5.04883 5.6222 5.67272 5 6.41981 5C7.16686 5 7.7908 5.62221 7.7908 6.40508C7.7908 7.18801 7.16722 7.8101 6.41981 7.8101C5.67241 7.8101 5.04883 7.18801 5.04883 6.40508Z" stroke="CurrentColor"></path>
                    <path d="M11.1045 6.40508C11.1045 5.62221 11.7284 5 12.4755 5C13.2229 5 13.8466 5.6222 13.8466 6.40508C13.8466 7.18789 13.2227 7.8101 12.4755 7.8101C11.7284 7.8101 11.1045 7.18794 11.1045 6.40508Z" stroke="CurrentColor"></path>
                    <path d="M19.9998 6.40514C19.9998 7.18797 19.3757 7.81016 18.6288 7.81016C17.8818 7.81016 17.2578 7.18794 17.2578 6.40508C17.2578 5.62211 17.8813 5 18.6288 5C19.3763 5 19.9998 5.62215 19.9998 6.40514Z" stroke="CurrentColor"></path>
                    <path d="M7.74249 12.5097C7.74249 13.2926 7.11849 13.9147 6.37133 13.9147C5.62411 13.9147 5 13.2926 5 12.5097C5 11.7267 5.62419 11.1044 6.37133 11.1044C7.11842 11.1044 7.74249 11.7266 7.74249 12.5097Z" stroke="CurrentColor"></path>
                    <path d="M13.7976 12.5097C13.7976 13.2927 13.1736 13.9147 12.4266 13.9147C11.6795 13.9147 11.0557 13.2927 11.0557 12.5097C11.0557 11.7265 11.6793 11.1044 12.4266 11.1044C13.1741 11.1044 13.7976 11.7265 13.7976 12.5097Z" stroke="CurrentColor"></path>
                    <path d="M19.9516 12.5097C19.9516 13.2927 19.328 13.9147 18.5807 13.9147C17.8329 13.9147 17.209 13.2925 17.209 12.5097C17.209 11.7268 17.8332 11.1044 18.5807 11.1044C19.3279 11.1044 19.9516 11.7265 19.9516 12.5097Z" stroke="CurrentColor"></path>
                    <path d="M5.04297 18.5947C5.04297 17.8118 5.66709 17.1896 6.4143 17.1896C7.16137 17.1896 7.78523 17.8116 7.78523 18.5947C7.78523 19.3778 7.16139 19.9997 6.4143 19.9997C5.66714 19.9997 5.04297 19.3773 5.04297 18.5947Z" stroke="CurrentColor"></path>
                    <path d="M11.0986 18.5947C11.0986 17.8118 11.7227 17.1896 12.47 17.1896C13.2169 17.1896 13.8409 17.8117 13.8409 18.5947C13.8409 19.3778 13.2169 19.9997 12.47 19.9997C11.7225 19.9997 11.0986 19.3774 11.0986 18.5947Z" stroke="CurrentColor"></path>
                    <path d="M17.252 18.5947C17.252 17.8117 17.876 17.1896 18.6229 17.1896C19.3699 17.1896 19.9939 17.8117 19.9939 18.5947C19.9939 19.3778 19.3702 19.9997 18.6229 19.9997C17.876 19.9997 17.252 19.3774 17.252 18.5947Z" stroke="CurrentColor"></path>
                  </svg>
                </a>
                <a href="#" class="btn-view list<?= $view==='list' ? ' active' : '' ?>" id="view-list">
                  <svg width="25" height="25" viewBox="0 0 25 25" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M19.7016 18.3317H9.00246C8.5615 18.3317 8.2041 17.9743 8.2041 17.5333C8.2041 17.0923 8.5615 16.7349 9.00246 16.7349H19.7013C20.1423 16.7349 20.4997 17.0923 20.4997 17.5333C20.4997 17.9743 20.1426 18.3317 19.7016 18.3317Z" fill="CurrentColor"></path>
                    <path d="M19.7016 13.3203H9.00246C8.5615 13.3203 8.2041 12.9629 8.2041 12.5219C8.2041 12.081 8.5615 11.7236 9.00246 11.7236H19.7013C20.1423 11.7236 20.4997 12.081 20.4997 12.5219C20.5 12.9629 20.1426 13.3203 19.7016 13.3203Z" fill="CurrentColor"></path>
                    <path d="M19.7016 8.30919H9.00246C8.5615 8.30919 8.2041 7.95179 8.2041 7.51083C8.2041 7.06986 8.5615 6.71246 9.00246 6.71246H19.7013C20.1423 6.71246 20.4997 7.06986 20.4997 7.51083C20.4997 7.95179 20.1426 8.30919 19.7016 8.30919Z" fill="CurrentColor"></path>
                    <path d="M5.5722 8.64465C6.16436 8.64465 6.6444 8.16461 6.6444 7.57245C6.6444 6.98029 6.16436 6.50024 5.5722 6.50024C4.98004 6.50024 4.5 6.98029 4.5 7.57245C4.5 8.16461 4.98004 8.64465 5.5722 8.64465Z" fill="CurrentColor"></path>
                    <path d="M5.5722 13.5942C6.16436 13.5942 6.6444 13.1141 6.6444 12.522C6.6444 11.9298 6.16436 11.4498 5.5722 11.4498C4.98004 11.4498 4.5 11.9298 4.5 12.522C4.5 13.1141 4.98004 13.5942 5.5722 13.5942Z" fill="CurrentColor"></path>
                    <path d="M5.5722 18.5438C6.16436 18.5438 6.6444 18.0637 6.6444 17.4716C6.6444 16.8794 6.16436 16.3994 5.5722 16.3994C4.98004 16.3994 4.5 16.8794 4.5 17.4716C4.5 18.0637 4.98004 18.5438 5.5722 18.5438Z" fill="CurrentColor"></path>
                  </svg>
                </a>

                <!-- Show (limit) -->
                <div class="wd-find-select flex gap-8">
                  <div class="group-select">
                    <div class="nice-select" id="nice-limit" tabindex="0">
                      <span class="current">Show: <?= (int)$limit ?></span>
                      <ul class="list style">
                        <?php foreach ([10,30,50,100] as $opt): ?>
                          <li data-value="<?= $opt ?>" class="option<?= $limit==$opt ? ' selected' : '' ?>">Show: <?= $opt ?></li>
                        <?php endforeach; ?>
                      </ul>
                    </div>
                  </div>

                  <!-- Sort -->
                  <div class="group-select">
                    <div class="nice-select" id="nice-sort" tabindex="0">
                      <span class="current">
                        <?php
                          echo $sort==='price-asc' ? 'Low to high' : ($sort==='price-desc' ? 'High to low' : 'Default order');
                        ?>
                      </span>
                      <ul class="list style">
                        <li data-value="newest" class="option<?= $sort==='newest' ? ' selected' : '' ?>">Default order</li>
                        <li data-value="newest" class="option">All</li>
                        <li data-value="price-asc" class="option<?= $sort==='price-asc' ? ' selected' : '' ?>">Low to high</li>
                        <li data-value="price-desc" class="option<?= $sort==='price-desc' ? ' selected' : '' ?>">High to low</li>
                      </ul>
                    </div>
                  </div>
                </div>

              </div>
            </div>
          </div>
        </div>
        <!-- Floating sub-filter chip -->
   

        <!-- Hidden state for AJAX -->
        <form id="products-filter" class="d-none" autocomplete="off">
          <input type="hidden" name="category_id" id="category_id" value="<?= (int)$category_id ?>">
          <input type="hidden" name="view"        id="view"        value="<?= h($view) ?>">
          <input type="hidden" name="page"        id="page"        value="<?= (int)$page ?>">
          <input type="hidden" name="limit"       id="limit"       value="<?= (int)$limit ?>">
          <input type="hidden" name="sort"        id="sort-hidden" value="<?= h($sort) ?>">
        </form>

        <!-- RESULTS WRAP (AJAX target) -->
        <div id="listing-results-wrap">
          <?php render_products_fragment($conn, $products, $view, $page, $totalPages, $totalRows, $fromRow, $toRow); ?>
        </div>

      </div>
    </section>

    <?php include './includes/footer.php'; ?>
  </div>
</div>

<!-- JS -->
<script src="/app/js/jquery.min.js"></script>
<script src="/app/js/jquery.easing.js"></script>
<script src="/app/js/jquery.nice-select.min.js"></script>
<script src="/app/js/bootstrap.min.js"></script>
<script src="/app/js/plugin.js"></script>
<script src="/app/js/shortcodes.js"></script>
<script src="/app/js/main.js"></script>

<script>
(function(){
  // ===== Utilities =====
  function skeletonCards(count){
    var view = document.getElementById('view').value || 'grid';
    count = +count || 12;
    var html = '';
    if (view === 'list') {
      html += '<div class="sk-list">';
      for (var i=0;i<count;i++){
        html += '<div class="car-skeleton"><div class="sk-img"></div><div><span class="sk-title"></span><span class="sk-detail"></span><span class="sk-meta"></span></div></div>';
      }
      html += '</div>';
    } else {
      html += '<div class="sk-grid">';
      for (var j=0;j<count;j++){
        html += '<div class="car-skeleton"><div class="sk-img"></div><span class="sk-title"></span><span class="sk-detail"></span><span class="sk-meta"></span></div>';
      }
      html += '</div>';
    }
    return html;
  }

  function currentParams(){
    var f  = document.getElementById('products-filter');
    var fd = new FormData(f);
    if (!fd.get('page')) fd.set('page', '1');
    return new URLSearchParams(Array.from(fd.entries()));
  }

  function pushUrl(params){
    var url = window.location.pathname + '?' + params.toString();
    history.replaceState({}, '', url);
    var og = document.getElementById('meta-og-url'); if (og) og.setAttribute('content', window.location.origin + window.location.pathname + '?' + params.toString());
  }

  function setMetaAndTitles(catName, results, from, to){
    var title = catName ? ('/ מוצרים ' + catName) : 'רשימת מוצרים';
    document.title = title;
    var dyn = document.getElementById('dynamic-title'); if (dyn) dyn.textContent = title;
    var bc  = document.getElementById('breadcrumb-leaf'); if (bc) bc.textContent = title;
    var h2  = document.getElementById('page-title'); if (h2) h2.textContent = catName ? catName : 'קטלוג מוצרים בר שרבלה';
    var desc= catName ? ('מוצרים בקטגוריית ' + catName) : 'מוצרים למכירה לפי קטגוריה';
    var keys= catName ? (catName + ', מוצרים, קטגוריות, מחירים') : 'מוצרים, קטגוריות, מחירים';
    var md  = document.getElementById('meta-desc'); if (md) md.setAttribute('content', desc);
    var mk  = document.getElementById('meta-keys'); if (mk) mk.setAttribute('content', keys);
    var ogt = document.getElementById('meta-og-title'); if (ogt) ogt.setAttribute('content', title);
    var ogd = document.getElementById('meta-og-desc'); if (ogd) ogd.setAttribute('content', desc);
    if (typeof results==='number' && typeof from==='number' && typeof to==='number') {
      var st = document.getElementById('showing-text');
      if (st) st.textContent = 'Showing ' + from + '–' + to + ' of ' + results + ' results';
      var cnt = document.getElementById('results-counter');
      if (cnt) cnt.textContent = results;
    }
  }

  function showChip(catName){
    var bar = document.getElementById('floating-subfilter');
    var lab = document.getElementById('chip-label');
    if (!bar || !lab) return;
    if (catName) {
      lab.textContent = 'קטגוריה: ' + catName;
      bar.classList.add('show');
    } else {
      bar.classList.remove('show');
    }
  }

  function bindPagination(){
    document.querySelectorAll('.ajax-page-link').forEach(function(a){
      a.addEventListener('click', function(e){
        e.preventDefault();
        var p = this.getAttribute('data-page');
        if (!p) return;
        document.getElementById('page').value = p;
        loadListings(false);
        // smooth scroll up to cube buttons
        var tabs = document.getElementById('cat-tabs');
        if (tabs) window.scrollTo({ top: tabs.offsetTop - 20, behavior: 'smooth' });
      });
    });
  }

  // ===== Main loader =====
  function loadListings(){
    var wrap = document.getElementById('listing-results-wrap');
    var params = currentParams();
    pushUrl(params);

    wrap.innerHTML = skeletonCards(document.getElementById('limit').value || 12);

    fetch(window.location.pathname + '?ajax=1&' + params.toString(), {
      headers: { 'X-Requested-With':'XMLHttpRequest' }
    }).then(function(r){ return r.text(); })
     .then(function(html){
        wrap.innerHTML = html;

        // Update view class
        var view = document.getElementById('view').value;
        var listDiv = wrap.querySelector('.list-car-list-1');
        if (listDiv) listDiv.className = 'list-car-list-1' + (view==='grid' ? ' list-car-grid-1' : '');

        // Update showing text & counters from meta
        var meta = wrap.querySelector('.listing-metadata');
        if (meta){
          var total = +meta.getAttribute('data-total') || 0;
          var from  = +meta.getAttribute('data-from')  || 0;
          var to    = +meta.getAttribute('data-to')    || 0;
          setMetaAndTitles(getSelectedCategoryName(), total, from, to);
        }

        bindPagination();
     });
  }

  function clearActiveCubes(){
    document.querySelectorAll('.cat-card').forEach(function(b){ b.classList.remove('active'); });
  }
  function getSelectedCategoryName(){
    var cid = document.getElementById('category_id').value;
    var btn = document.querySelector('.cat-card[data-category-id="'+cid+'"]');
    return btn ? (btn.getAttribute('data-category-name') || '') : '';
  }

  // ===== Bind UI =====
  // Icon cubes (category change)
  document.querySelectorAll('.cat-card').forEach(function(btn){
    btn.addEventListener('click', function(){
      clearActiveCubes();
      this.classList.add('active');

      var cid   = this.getAttribute('data-category-id') || '0';
      var cname = this.getAttribute('data-category-name') || '';
      document.getElementById('category_id').value = cid;
      document.getElementById('page').value = '1';

      // update titles/meta + chip
      setMetaAndTitles(cname);

      // load + scroll up to buttons
      loadListings();
      var tabs = document.getElementById('cat-tabs');
      if (tabs) window.scrollTo({ top: tabs.offsetTop - 20, behavior: 'smooth' });
    });
  });

  // Chip clear
  var chipClear = document.getElementById('chip-clear');
  if (chipClear) chipClear.addEventListener('click', function(){
    document.getElementById('category_id').value = '0';
    document.getElementById('page').value = '1';
    setMetaAndTitles('');
    clearActiveCubes();
    // mark the "All categories" cube active
    var allBtn = document.querySelector('.cat-card[data-category-id="0"]');
    if (allBtn) allBtn.classList.add('active');
    loadListings();
    var tabs = document.getElementById('cat-tabs');
    if (tabs) window.scrollTo({ top: tabs.offsetTop - 20, behavior: 'smooth' });
  });

  // View toggles
  var btnGrid = document.getElementById('view-grid');
  var btnList = document.getElementById('view-list');
  if (btnGrid) btnGrid.addEventListener('click', function(e){ e.preventDefault();
    document.getElementById('view').value = 'grid';
    btnGrid.classList.add('active'); btnList.classList.remove('active');
    document.getElementById('page').value = '1';
    loadListings();
  });
  if (btnList) btnList.addEventListener('click', function(e){ e.preventDefault();
    document.getElementById('view').value = 'list';
    btnList.classList.add('active'); btnGrid.classList.remove('active');
    document.getElementById('page').value = '1';
    loadListings();
  });

  // Nice-select (Show: limit)
  var niceLimit = document.getElementById('nice-limit');
  if (niceLimit){
    niceLimit.querySelectorAll('.option').forEach(function(opt){
      opt.addEventListener('click', function(){
        var val = this.getAttribute('data-value');
        if (!val) return;
        niceLimit.querySelectorAll('.option').forEach(function(o){ o.classList.remove('selected'); });
        this.classList.add('selected');
        niceLimit.querySelector('.current').textContent = 'Show: ' + val;
        document.getElementById('limit').value = val;
        document.getElementById('page').value = '1';
        loadListings();
      });
    });
  }

  // Nice-select (Sort)
  var niceSort = document.getElementById('nice-sort');
  if (niceSort){
    niceSort.querySelectorAll('.option').forEach(function(opt){
      opt.addEventListener('click', function(){
        var val = this.getAttribute('data-value') || 'newest';
        niceSort.querySelectorAll('.option').forEach(function(o){ o.classList.remove('selected'); });
        this.classList.add('selected');
        // label
        var label = this.textContent.trim();
        niceSort.querySelector('.current').textContent = label;
        // map to server sort values (newest, price-asc, price-desc) already passed
        document.getElementById('sort-hidden').value = val;
        document.getElementById('page').value = '1';
        loadListings();
      });
    });
  }

  // Show initial chip if needed
  if (document.getElementById('category_id').value !== '0') {
    showChip(getSelectedCategoryName());
  } else {
    // ensure "All categories" cube is active when none chosen
    var allBtn = document.querySelector('.cat-card[data-category-id="0"]');
    if (allBtn) allBtn.classList.add('active');
  }

  // On initial render, pagination is server-rendered → enable ajax nav
  bindPagination();
})();
</script>
</body>
</html>
